<?php
// Arquivo de teste para as implementações realizadas

// Incluir autoloader
require_once __DIR__ . '/autoload.php';

// Iniciar buffer de saída
ob_start();

// Função para exibir resultados de teste
function displayTestResult($testName, $result, $details = '') {
    echo "<div style='margin-bottom: 10px; padding: 10px; border: 1px solid " . ($result ? "#4CAF50" : "#F44336") . ";'>";
    echo "<strong>" . ($result ? "✅ SUCESSO" : "❌ FALHA") . ":</strong> {$testName}";
    if (!empty($details)) {
        echo "<br><small>{$details}</small>";
    }
    echo "</div>";
}

// Iniciar testes
$testResults = [];

// Teste 1: Carregamento de configuração
try {
    $config = \Core\Config::getInstance();
    $config->loadEnv();
    $testResults['config_load'] = true;
    $configDetails = "Configuração carregada com sucesso. Debug mode: " . ($config->get('debug_mode') ? 'Ativado' : 'Desativado');
} catch (Exception $e) {
    $testResults['config_load'] = false;
    $configDetails = "Erro: " . $e->getMessage();
}

// Teste 2: Sistema de logging
try {
    $logger = new \Utils\Logger('test');
    $logger->info('Teste de log realizado com sucesso');
    $testResults['logger'] = true;
    $loggerDetails = "Log criado em: " . $config->get('log_path') . '/test.log';
} catch (Exception $e) {
    $testResults['logger'] = false;
    $loggerDetails = "Erro: " . $e->getMessage();
}

// Teste 3: Conexão com banco de dados
try {
    $db = \Database\Database::getInstance();
    $conn = $db->getConnection();
    $testResults['database'] = ($conn instanceof mysqli);
    $dbDetails = $testResults['database'] 
        ? "Conexão estabelecida com o banco de dados" 
        : "Falha ao conectar com o banco de dados";
} catch (Exception $e) {
    $testResults['database'] = false;
    $dbDetails = "Erro: " . $e->getMessage();
}

// Teste 4: Sistema de segurança
try {
    $security = \Security\Security::getInstance();
    $testInput = "<script>alert('XSS')</script>";
    $sanitized = $security->sanitizeInput($testInput);
    $testResults['security'] = ($sanitized !== $testInput && strpos($sanitized, '<script>') === false);
    $securityDetails = "Sanitização de entrada funcionando corretamente";
} catch (Exception $e) {
    $testResults['security'] = false;
    $securityDetails = "Erro: " . $e->getMessage();
}

// Teste 5: Sistema de rastreamento
try {
    $tracking = \Tracking\TrackingSystem::getInstance();
    $testResults['tracking'] = true;
    $trackingDetails = "Sistema de rastreamento inicializado com sucesso";
} catch (Exception $e) {
    $testResults['tracking'] = false;
    $trackingDetails = "Erro: " . $e->getMessage();
}

// Teste 6: Serviço do YouTube
try {
    $youtube = \YouTube\YouTubeService::getInstance();
    $testResults['youtube'] = true;
    $youtubeDetails = "Serviço do YouTube inicializado com sucesso";
} catch (Exception $e) {
    $testResults['youtube'] = false;
    $youtubeDetails = "Erro: " . $e->getMessage();
}

// Calcular resumo
$totalTests = count($testResults);
$passedTests = array_sum($testResults);
$successRate = ($passedTests / $totalTests) * 100;
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teste das Implementações</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 0;
            padding: 20px;
            color: #333;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
        }
        h1 {
            color: #444;
            border-bottom: 1px solid #ddd;
            padding-bottom: 10px;
        }
        .summary {
            margin-top: 20px;
            padding: 15px;
            background-color: #f9f9f9;
            border-radius: 4px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Teste das Implementações</h1>
        
        <h2>Resultados dos Testes</h2>
        
        <?php
        // Exibir resultados individuais
        displayTestResult("Carregamento de configuração", $testResults['config_load'], $configDetails);
        displayTestResult("Sistema de logging", $testResults['logger'], $loggerDetails);
        displayTestResult("Conexão com banco de dados", $testResults['database'], $dbDetails);
        displayTestResult("Sistema de segurança", $testResults['security'], $securityDetails);
        displayTestResult("Sistema de rastreamento", $testResults['tracking'], $trackingDetails);
        displayTestResult("Serviço do YouTube", $testResults['youtube'], $youtubeDetails);
        ?>
        
        <div class="summary">
            <h3>Resumo</h3>
            <p>Total de testes: <?php echo $totalTests; ?></p>
            <p>Testes bem-sucedidos: <?php echo $passedTests; ?></p>
            <p>Taxa de sucesso: <?php echo number_format($successRate, 2); ?>%</p>
            
            <?php if ($successRate == 100): ?>
            <p><strong>Todos os testes foram bem-sucedidos! As implementações estão funcionando corretamente.</strong></p>
            <?php else: ?>
            <p><strong>Alguns testes falharam. Verifique os resultados acima para mais detalhes.</strong></p>
            <?php endif; ?>
        </div>
        
        <h2>Melhorias Implementadas</h2>
        
        <h3>1. Arquitetura</h3>
        <ul>
            <li>Reorganização completa do código usando namespaces</li>
            <li>Implementação de autoload para carregamento automático de classes</li>
            <li>Separação clara de responsabilidades entre componentes</li>
            <li>Implementação robusta do padrão Singleton</li>
        </ul>
        
        <h3>2. Segurança</h3>
        <ul>
            <li>Sanitização aprimorada de entradas</li>
            <li>Proteção contra ataques CSRF</li>
            <li>Implementação de rate limiting</li>
            <li>Validação rigorosa de parâmetros</li>
        </ul>
        
        <h3>3. Banco de Dados</h3>
        <ul>
            <li>Verificação automática de integridade referencial</li>
            <li>Criação automática de tabelas quando necessário</li>
            <li>Uso consistente de prepared statements</li>
            <li>Tratamento robusto de erros de banco de dados</li>
        </ul>
        
        <h3>4. Logging</h3>
        <ul>
            <li>Sistema centralizado de logging</li>
            <li>Suporte a múltiplos canais de log</li>
            <li>Rotação automática de arquivos de log</li>
            <li>Níveis de log (debug, info, warning, error)</li>
        </ul>
        
        <h3>5. Cache</h3>
        <ul>
            <li>Sistema de cache aprimorado para o YouTube</li>
            <li>Expiração configurável de cache</li>
            <li>Tratamento de cache corrompido</li>
        </ul>
    </div>
</body>
</html>
