<?php
namespace Tracking;

use Core\Config;
use Utils\Logger;
use Database\Database;
use Security\Security;

/**
 * Classe TrackingSystem - Sistema unificado de rastreamento
 * 
 * Versão melhorada que:
 * - Utiliza namespaces para organização
 * - Implementa padrão Singleton de forma mais robusta
 * - Melhora o tratamento de erros e validação
 * - Implementa verificação de integridade referencial
 */

class TrackingSystem {
    private static $instance = null;
    private $db;
    private $security;
    private $logger;
    private $config;
    
    /**
     * Construtor privado (padrão Singleton)
     */
    private function __construct() {
        $this->db = Database::getInstance();
        $this->security = Security::getInstance();
        $this->logger = new Logger('tracking');
        $this->config = Config::getInstance();
        
        // Verificar e criar tabelas necessárias
        $this->initializeTables();
    }
    
    /**
     * Previne clonagem do objeto (padrão Singleton)
     */
    private function __clone() {}
    
    /**
     * Previne desserialização do objeto (padrão Singleton)
     */
    private function __wakeup() {}
    
    /**
     * Obtém a instância única da classe
     * 
     * @return TrackingSystem
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        
        return self::$instance;
    }
    
    /**
     * Inicializa as tabelas necessárias para o sistema de rastreamento
     */
    private function initializeTables() {
        // Tabela de links
        $schema = "CREATE TABLE IF NOT EXISTS links (
            id INT(11) NOT NULL AUTO_INCREMENT,
            url VARCHAR(255) NOT NULL,
            title VARCHAR(100) NOT NULL,
            description VARCHAR(255) DEFAULT NULL,
            active TINYINT(1) DEFAULT 1,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            INDEX (active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('links', $schema);
        
        // Tabela de cliques em links
        $schema = "CREATE TABLE IF NOT EXISTS link_clicks (
            id INT(11) NOT NULL AUTO_INCREMENT,
            link_id INT(11) NOT NULL,
            visitor_ip VARCHAR(45) NOT NULL,
            visitor_device_type VARCHAR(20) NOT NULL,
            click_date DATETIME NOT NULL,
            PRIMARY KEY (id),
            INDEX (link_id),
            INDEX (click_date),
            CONSTRAINT link_clicks_ibfk_1 FOREIGN KEY (link_id) REFERENCES links (id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('link_clicks', $schema);
        
        // Tabela de teste para cliques em links (sem restrição de chave estrangeira)
        $schema = "CREATE TABLE IF NOT EXISTS test_link_clicks (
            id INT(11) NOT NULL AUTO_INCREMENT,
            link_id INT(11) NOT NULL,
            visitor_ip VARCHAR(45) NOT NULL,
            visitor_device_type VARCHAR(20) NOT NULL,
            click_date DATETIME NOT NULL,
            PRIMARY KEY (id),
            INDEX (link_id),
            INDEX (click_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('test_link_clicks', $schema);
        
        // Tabela de banners
        $schema = "CREATE TABLE IF NOT EXISTS banners (
            id INT(11) NOT NULL AUTO_INCREMENT,
            image_url VARCHAR(255) NOT NULL,
            link_url VARCHAR(255) NOT NULL,
            title VARCHAR(100) NOT NULL,
            active TINYINT(1) DEFAULT 1,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            INDEX (active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('banners', $schema);
        
        // Tabela de cliques em banners
        $schema = "CREATE TABLE IF NOT EXISTS banner_clicks (
            id INT(11) NOT NULL AUTO_INCREMENT,
            banner_id INT(11) NOT NULL,
            visitor_ip VARCHAR(45) NOT NULL,
            visitor_device_type VARCHAR(20) NOT NULL,
            click_date DATETIME NOT NULL,
            PRIMARY KEY (id),
            INDEX (banner_id),
            INDEX (click_date),
            CONSTRAINT banner_clicks_ibfk_1 FOREIGN KEY (banner_id) REFERENCES banners (id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('banner_clicks', $schema);
        
        // Tabela de teste para cliques em banners (sem restrição de chave estrangeira)
        $schema = "CREATE TABLE IF NOT EXISTS test_banner_clicks (
            id INT(11) NOT NULL AUTO_INCREMENT,
            banner_id INT(11) NOT NULL,
            visitor_ip VARCHAR(45) NOT NULL,
            visitor_device_type VARCHAR(20) NOT NULL,
            click_date DATETIME NOT NULL,
            PRIMARY KEY (id),
            INDEX (banner_id),
            INDEX (click_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('test_banner_clicks', $schema);
        
        // Tabela de redes sociais
        $schema = "CREATE TABLE IF NOT EXISTS socials (
            id INT(11) NOT NULL AUTO_INCREMENT,
            platform VARCHAR(50) NOT NULL,
            url VARCHAR(255) NOT NULL,
            icon VARCHAR(50) DEFAULT NULL,
            active TINYINT(1) DEFAULT 1,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            INDEX (active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('socials', $schema);
        
        // Tabela de cliques em redes sociais
        $schema = "CREATE TABLE IF NOT EXISTS social_clicks (
            id INT(11) NOT NULL AUTO_INCREMENT,
            social_id INT(11) NOT NULL,
            visitor_ip VARCHAR(45) NOT NULL,
            visitor_device_type VARCHAR(20) NOT NULL,
            click_date DATETIME NOT NULL,
            PRIMARY KEY (id),
            INDEX (social_id),
            INDEX (click_date),
            CONSTRAINT social_clicks_ibfk_1 FOREIGN KEY (social_id) REFERENCES socials (id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('social_clicks', $schema);
        
        // Tabela de teste para cliques em redes sociais (sem restrição de chave estrangeira)
        $schema = "CREATE TABLE IF NOT EXISTS test_social_clicks (
            id INT(11) NOT NULL AUTO_INCREMENT,
            social_id INT(11) NOT NULL,
            visitor_ip VARCHAR(45) NOT NULL,
            visitor_device_type VARCHAR(20) NOT NULL,
            click_date DATETIME NOT NULL,
            PRIMARY KEY (id),
            INDEX (social_id),
            INDEX (click_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('test_social_clicks', $schema);
        
        // Tabela de visualizações de página
        $schema = "CREATE TABLE IF NOT EXISTS page_views (
            id INT(11) NOT NULL AUTO_INCREMENT,
            page_url VARCHAR(255) DEFAULT NULL,
            visitor_ip VARCHAR(45) NOT NULL,
            visitor_device_type VARCHAR(20) NOT NULL,
            referer VARCHAR(255) DEFAULT NULL,
            view_date DATETIME NOT NULL,
            PRIMARY KEY (id),
            INDEX (view_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('page_views', $schema);
        
        // Tabela de visualizações de seção
        $schema = "CREATE TABLE IF NOT EXISTS section_views (
            id INT(11) NOT NULL AUTO_INCREMENT,
            section_name VARCHAR(50) NOT NULL,
            visitor_ip VARCHAR(45) NOT NULL,
            visitor_device_type VARCHAR(20) NOT NULL,
            view_date DATETIME NOT NULL,
            PRIMARY KEY (id),
            INDEX (section_name),
            INDEX (view_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('section_views', $schema);
        
        // Tabela de vídeos
        $schema = "CREATE TABLE IF NOT EXISTS videos (
            id INT(11) NOT NULL AUTO_INCREMENT,
            youtube_id VARCHAR(20) NOT NULL,
            title VARCHAR(100) NOT NULL,
            description TEXT DEFAULT NULL,
            active TINYINT(1) DEFAULT 1,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            INDEX (active)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('videos', $schema);
        
        // Tabela de eventos de vídeo
        $schema = "CREATE TABLE IF NOT EXISTS video_events (
            id INT(11) NOT NULL AUTO_INCREMENT,
            video_id INT(11) NOT NULL,
            action VARCHAR(20) NOT NULL,
            position INT(11) DEFAULT 0,
            visitor_ip VARCHAR(45) NOT NULL,
            visitor_device_type VARCHAR(20) NOT NULL,
            event_date DATETIME NOT NULL,
            PRIMARY KEY (id),
            INDEX (video_id),
            INDEX (action),
            INDEX (event_date),
            CONSTRAINT video_events_ibfk_1 FOREIGN KEY (video_id) REFERENCES videos (id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('video_events', $schema);
        
        // Tabela de teste para eventos de vídeo (sem restrição de chave estrangeira)
        $schema = "CREATE TABLE IF NOT EXISTS test_video_events (
            id INT(11) NOT NULL AUTO_INCREMENT,
            video_id INT(11) NOT NULL,
            action VARCHAR(20) NOT NULL,
            position INT(11) DEFAULT 0,
            visitor_ip VARCHAR(45) NOT NULL,
            visitor_device_type VARCHAR(20) NOT NULL,
            event_date DATETIME NOT NULL,
            PRIMARY KEY (id),
            INDEX (video_id),
            INDEX (action),
            INDEX (event_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('test_video_events', $schema);
        
        // Tabela de engajamento
        $schema = "CREATE TABLE IF NOT EXISTS engagement (
            id INT(11) NOT NULL AUTO_INCREMENT,
            engagement_type VARCHAR(20) NOT NULL,
            engagement_value INT(11) NOT NULL,
            page_url VARCHAR(255) DEFAULT NULL,
            visitor_ip VARCHAR(45) NOT NULL,
            visitor_device_type VARCHAR(20) NOT NULL,
            event_date DATETIME NOT NULL,
            PRIMARY KEY (id),
            INDEX (engagement_type),
            INDEX (event_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;";
        $this->db->createTableIfNotExists('engagement', $schema);
    }
    
    /**
     * Registra um clique em link, banner ou rede social
     * 
     * @param int $id ID do item clicado
     * @param string $type Tipo do item (link, banner, social, video)
     * @return bool Sucesso ou falha
     */
    public function trackClick($id, $type = 'link') {
        // Sanitizar e validar parâmetros
        $id = $this->security->sanitizeInt($id);
        $type = $this->security->sanitizeInput($type);
        
        if ($id === false || $id <= 0) {
            $this->logger->error("ID inválido recebido", [
                'id' => $id,
                'type' => $type
            ]);
            return false;
        }
        
        try {
            // Obter informações do visitante
            $ip = $this->security->getClientIp();
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
            $is_mobile = preg_match('/(android|iphone|ipad|mobile)/i', $user_agent) ? 'mobile' : 'desktop';
            $timestamp = date('Y-m-d H:i:s');
            
            // Determinar a tabela e a tabela de referência com base no tipo
            $table = '';
            $ref_table = '';
            $id_column = '';
            
            switch ($type) {
                case 'link':
                    $table = 'link_clicks';
                    $ref_table = 'links';
                    $id_column = 'link_id';
                    break;
                case 'social':
                    $table = 'social_clicks';
                    $ref_table = 'socials';
                    $id_column = 'social_id';
                    break;
                case 'banner':
                    $table = 'banner_clicks';
                    $ref_table = 'banners';
                    $id_column = 'banner_id';
                    break;
                case 'video':
                    $table = 'video_events';
                    $ref_table = 'videos';
                    $id_column = 'video_id';
                    break;
                default:
                    $table = 'link_clicks';
                    $ref_table = 'links';
                    $id_column = 'link_id';
            }
            
            // Verificar se o ID existe na tabela de referência
            $id_exists = $this->db->recordExists($ref_table, 'id', $id);
            
            // Se o ID não existe, usar tabela de teste
            if (!$id_exists) {
                $this->logger->info("ID {$id} não existe na tabela {$ref_table}, usando tabela de teste", [
                    'id' => $id,
                    'type' => $type
                ]);
                $table = 'test_' . $table;
                
                // Para eventos de vídeo, ajustar nome da tabela
                if ($type === 'video') {
                    $table = 'test_video_events';
                }
            }
            
            // Inserir o registro de clique
            if ($type === 'video') {
                $insert_sql = "INSERT INTO {$table} ({$id_column}, action, position, visitor_ip, visitor_device_type, event_date) 
                              VALUES (?, 'click', 0, ?, ?, ?)";
                $insert_result = $this->security->executeQuery($insert_sql, [$id, $ip, $is_mobile, $timestamp], 'isss');
            } else {
                $insert_sql = "INSERT INTO {$table} ({$id_column}, visitor_ip, visitor_device_type, click_date) VALUES (?, ?, ?, ?)";
                $insert_result = $this->security->executeQuery($insert_sql, [$id, $ip, $is_mobile, $timestamp], 'isss');
            }
            
            if ($insert_result) {
                $this->logger->info("Clique em {$type} registrado com sucesso", [
                    'id' => $id,
                    'table' => $table,
                    'ip' => $ip,
                    'device' => $is_mobile
                ]);
                return true;
            } else {
                $this->logger->error("Falha ao registrar clique em {$type}", [
                    'id' => $id,
                    'table' => $table
                ]);
                return false;
            }
            
        } catch (\Exception $e) {
            $this->logger->error("Exceção ao rastrear clique em {$type}: " . $e->getMessage(), [
                'id' => $id,
                'type' => $type,
                'exception' => get_class($e)
            ]);
            return false;
        }
    }
    
    /**
     * Registra uma visualização de página
     * 
     * @param string $url URL da página
     * @param string $referer Referer da visita
     * @return bool Sucesso ou falha
     */
    public function trackPageView($url = '', $referer = '') {
        try {
            // Sanitizar parâmetros
            $url = $this->security->sanitizeInput($url);
            if (empty($url)) {
                $url = $_SERVER['REQUEST_URI'] ?? '';
            }
            
            if (empty($referer)) {
                $referer = $_SERVER['HTTP_REFERER'] ?? '';
            }
            $referer = $this->security->sanitizeInput($referer);
            
            // Obter informações do visitante
            $ip = $this->security->getClientIp();
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
            $is_mobile = preg_match('/(android|iphone|ipad|mobile)/i', $user_agent) ? 'mobile' : 'desktop';
            $timestamp = date('Y-m-d H:i:s');
            
            // Inserir o registro de visualização
            $insert_sql = "INSERT INTO page_views (page_url, visitor_ip, visitor_device_type, referer, view_date) VALUES (?, ?, ?, ?, ?)";
            $insert_result = $this->security->executeQuery($insert_sql, [$url, $ip, $is_mobile, $referer, $timestamp], 'sssss');
            
            if ($insert_result) {
                $this->logger->info("Visualização de página registrada", [
                    'url' => $url,
                    'ip' => $ip,
                    'device' => $is_mobile
                ]);
                return true;
            } else {
                $this->logger->error("Falha ao registrar visualização de página", [
                    'url' => $url
                ]);
                return false;
            }
            
        } catch (\Exception $e) {
            $this->logger->error("Exceção ao rastrear visualização de página: " . $e->getMessage(), [
                'url' => $url ?? '',
                'exception' => get_class($e)
            ]);
            return false;
        }
    }
    
    /**
     * Registra uma visualização de seção
     * 
     * @param string $section Nome da seção
     * @return bool Sucesso ou falha
     */
    public function trackSectionView($section) {
        try {
            // Sanitizar parâmetros
            $section = $this->security->sanitizeInput($section);
            
            if (empty($section)) {
                $this->logger->error("Nome da seção não fornecido");
                return false;
            }
            
            // Obter informações do visitante
            $ip = $this->security->getClientIp();
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
            $is_mobile = preg_match('/(android|iphone|ipad|mobile)/i', $user_agent) ? 'mobile' : 'desktop';
            $timestamp = date('Y-m-d H:i:s');
            
            // Inserir o registro de visualização
            $insert_sql = "INSERT INTO section_views (section_name, visitor_ip, visitor_device_type, view_date) VALUES (?, ?, ?, ?)";
            $insert_result = $this->security->executeQuery($insert_sql, [$section, $ip, $is_mobile, $timestamp], 'ssss');
            
            if ($insert_result) {
                $this->logger->info("Visualização de seção registrada", [
                    'section' => $section,
                    'ip' => $ip,
                    'device' => $is_mobile
                ]);
                return true;
            } else {
                $this->logger->error("Falha ao registrar visualização de seção", [
                    'section' => $section
                ]);
                return false;
            }
            
        } catch (\Exception $e) {
            $this->logger->error("Exceção ao rastrear visualização de seção: " . $e->getMessage(), [
                'section' => $section ?? '',
                'exception' => get_class($e)
            ]);
            return false;
        }
    }
    
    /**
     * Registra um evento de vídeo (play, pause, complete)
     * 
     * @param int $video_id ID do vídeo
     * @param string $action Ação (play, pause, complete)
     * @param int $position Posição do vídeo em segundos
     * @return bool Sucesso ou falha
     */
    public function trackVideoEvent($video_id, $action, $position = 0) {
        try {
            // Sanitizar parâmetros
            $video_id = $this->security->sanitizeInt($video_id);
            $action = $this->security->sanitizeInput($action);
            $position = $this->security->sanitizeInt($position);
            
            if ($video_id === false || $video_id <= 0) {
                $this->logger->error("ID de vídeo inválido", [
                    'video_id' => $video_id
                ]);
                return false;
            }
            
            // Validar ação
            $valid_actions = ['play', 'pause', 'complete', 'progress'];
            if (!in_array($action, $valid_actions)) {
                $this->logger->error("Ação de vídeo inválida", [
                    'action' => $action
                ]);
                return false;
            }
            
            // Obter informações do visitante
            $ip = $this->security->getClientIp();
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
            $is_mobile = preg_match('/(android|iphone|ipad|mobile)/i', $user_agent) ? 'mobile' : 'desktop';
            $timestamp = date('Y-m-d H:i:s');
            
            // Verificar se o ID existe na tabela de referência
            $id_exists = $this->db->recordExists('videos', 'id', $video_id);
            
            // Determinar a tabela com base na existência do ID
            $table = $id_exists ? 'video_events' : 'test_video_events';
            
            // Inserir o registro de evento
            $insert_sql = "INSERT INTO {$table} (video_id, action, position, visitor_ip, visitor_device_type, event_date) 
                          VALUES (?, ?, ?, ?, ?, ?)";
            $insert_result = $this->security->executeQuery(
                $insert_sql, 
                [$video_id, $action, $position, $ip, $is_mobile, $timestamp], 
                'isiiss'
            );
            
            if ($insert_result) {
                $this->logger->info("Evento de vídeo registrado", [
                    'video_id' => $video_id,
                    'action' => $action,
                    'position' => $position,
                    'ip' => $ip,
                    'device' => $is_mobile
                ]);
                return true;
            } else {
                $this->logger->error("Falha ao registrar evento de vídeo", [
                    'video_id' => $video_id,
                    'action' => $action
                ]);
                return false;
            }
            
        } catch (\Exception $e) {
            $this->logger->error("Exceção ao rastrear evento de vídeo: " . $e->getMessage(), [
                'video_id' => $video_id ?? 0,
                'action' => $action ?? '',
                'exception' => get_class($e)
            ]);
            return false;
        }
    }
    
    /**
     * Registra um evento de engajamento (scroll, tempo na página)
     * 
     * @param string $type Tipo de engajamento (scroll, time)
     * @param mixed $value Valor do engajamento (porcentagem de scroll ou tempo em segundos)
     * @param string $page_url URL da página
     * @return bool Sucesso ou falha
     */
    public function trackEngagement($type, $value, $page_url = '') {
        try {
            // Sanitizar parâmetros
            $type = $this->security->sanitizeInput($type);
            $value = $this->security->sanitizeInt($value);
            $page_url = $this->security->sanitizeInput($page_url);
            
            if (empty($page_url)) {
                $page_url = $_SERVER['REQUEST_URI'] ?? '';
            }
            
            // Validar tipo
            $valid_types = ['scroll', 'time'];
            if (!in_array($type, $valid_types)) {
                $this->logger->error("Tipo de engajamento inválido", [
                    'type' => $type
                ]);
                return false;
            }
            
            // Obter informações do visitante
            $ip = $this->security->getClientIp();
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
            $is_mobile = preg_match('/(android|iphone|ipad|mobile)/i', $user_agent) ? 'mobile' : 'desktop';
            $timestamp = date('Y-m-d H:i:s');
            
            // Inserir o registro de engajamento
            $insert_sql = "INSERT INTO engagement (engagement_type, engagement_value, page_url, visitor_ip, visitor_device_type, event_date) 
                          VALUES (?, ?, ?, ?, ?, ?)";
            $insert_result = $this->security->executeQuery(
                $insert_sql, 
                [$type, $value, $page_url, $ip, $is_mobile, $timestamp], 
                'sissss'
            );
            
            if ($insert_result) {
                $this->logger->info("Engajamento registrado", [
                    'type' => $type,
                    'value' => $value,
                    'page_url' => $page_url,
                    'ip' => $ip,
                    'device' => $is_mobile
                ]);
                return true;
            } else {
                $this->logger->error("Falha ao registrar engajamento", [
                    'type' => $type,
                    'value' => $value
                ]);
                return false;
            }
            
        } catch (\Exception $e) {
            $this->logger->error("Exceção ao rastrear engajamento: " . $e->getMessage(), [
                'type' => $type ?? '',
                'value' => $value ?? 0,
                'exception' => get_class($e)
            ]);
            return false;
        }
    }
    
    /**
     * Obtém estatísticas de cliques por período
     * 
     * @param string $type Tipo de clique (link, banner, social)
     * @param string $period Período (day, week, month, year)
     * @return array Array com estatísticas
     */
    public function getClickStats($type = 'link', $period = 'week') {
        try {
            // Determinar a tabela com base no tipo
            $table = '';
            switch ($type) {
                case 'link':
                    $table = 'link_clicks';
                    break;
                case 'social':
                    $table = 'social_clicks';
                    break;
                case 'banner':
                    $table = 'banner_clicks';
                    break;
                default:
                    $table = 'link_clicks';
            }
            
            // Determinar a data de início com base no período
            $start_date = '';
            switch ($period) {
                case 'day':
                    $start_date = date('Y-m-d 00:00:00');
                    break;
                case 'week':
                    $start_date = date('Y-m-d 00:00:00', strtotime('-7 days'));
                    break;
                case 'month':
                    $start_date = date('Y-m-d 00:00:00', strtotime('-30 days'));
                    break;
                case 'year':
                    $start_date = date('Y-m-d 00:00:00', strtotime('-365 days'));
                    break;
                default:
                    $start_date = date('Y-m-d 00:00:00', strtotime('-7 days'));
            }
            
            // Consultar estatísticas
            $sql = "SELECT 
                        COUNT(*) as total_clicks,
                        COUNT(DISTINCT visitor_ip) as unique_visitors,
                        DATE(click_date) as click_day,
                        SUM(CASE WHEN visitor_device_type = 'mobile' THEN 1 ELSE 0 END) as mobile_clicks,
                        SUM(CASE WHEN visitor_device_type = 'desktop' THEN 1 ELSE 0 END) as desktop_clicks
                    FROM {$table}
                    WHERE click_date >= ?
                    GROUP BY click_day
                    ORDER BY click_day ASC";
            
            $result = $this->security->executeQuery($sql, [$start_date], 's');
            
            $stats = [];
            if ($result) {
                while ($row = $result->fetch_assoc()) {
                    $stats[] = $row;
                }
            }
            
            return $stats;
            
        } catch (\Exception $e) {
            $this->logger->error("Exceção ao obter estatísticas de cliques: " . $e->getMessage(), [
                'type' => $type,
                'period' => $period,
                'exception' => get_class($e)
            ]);
            return [];
        }
    }
    
    /**
     * Obtém estatísticas de visualizações de página por período
     * 
     * @param string $period Período (day, week, month, year)
     * @return array Array com estatísticas
     */
    public function getPageViewStats($period = 'week') {
        try {
            // Determinar a data de início com base no período
            $start_date = '';
            switch ($period) {
                case 'day':
                    $start_date = date('Y-m-d 00:00:00');
                    break;
                case 'week':
                    $start_date = date('Y-m-d 00:00:00', strtotime('-7 days'));
                    break;
                case 'month':
                    $start_date = date('Y-m-d 00:00:00', strtotime('-30 days'));
                    break;
                case 'year':
                    $start_date = date('Y-m-d 00:00:00', strtotime('-365 days'));
                    break;
                default:
                    $start_date = date('Y-m-d 00:00:00', strtotime('-7 days'));
            }
            
            // Consultar estatísticas
            $sql = "SELECT 
                        COUNT(*) as total_views,
                        COUNT(DISTINCT visitor_ip) as unique_visitors,
                        DATE(view_date) as view_day,
                        SUM(CASE WHEN visitor_device_type = 'mobile' THEN 1 ELSE 0 END) as mobile_views,
                        SUM(CASE WHEN visitor_device_type = 'desktop' THEN 1 ELSE 0 END) as desktop_views
                    FROM page_views
                    WHERE view_date >= ?
                    GROUP BY view_day
                    ORDER BY view_day ASC";
            
            $result = $this->security->executeQuery($sql, [$start_date], 's');
            
            $stats = [];
            if ($result) {
                while ($row = $result->fetch_assoc()) {
                    $stats[] = $row;
                }
            }
            
            return $stats;
            
        } catch (\Exception $e) {
            $this->logger->error("Exceção ao obter estatísticas de visualizações de página: " . $e->getMessage(), [
                'period' => $period,
                'exception' => get_class($e)
            ]);
            return [];
        }
    }
}
