<?php
// Arquivo para exibir múltiplos banners com efeitos hover
// Salve este arquivo como multi_banners.php na raiz do seu site

// Obter banners ativos
function getActiveBanners($conn, $limit = 3) {
    $banners = [];
    
    $sql = "SELECT * FROM banners WHERE is_active = 1 ORDER BY order_position ASC LIMIT ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $limit);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $banners[] = $row;
    }
    
    $stmt->close();
    
    return $banners;
}

// Função para obter o último vídeo do YouTube
function getLatestYoutubeVideo($conn) {
    // Obter configurações do YouTube
    $channel_id = '';
    $api_key = '';
    
    $sql = "SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('youtube_channel_id', 'youtube_api_key')";
    $result = $conn->query($sql);
    
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            if ($row['setting_key'] === 'youtube_channel_id') {
                $channel_id = $row['setting_value'];
            } elseif ($row['setting_key'] === 'youtube_api_key') {
                $api_key = $row['setting_value'];
            }
        }
    }
    
    // Verificar se as configurações estão definidas
    if (empty($channel_id) || empty($api_key)) {
        return [
            'success' => false,
            'message' => 'Configurações do YouTube não encontradas'
        ];
    }
    
    // Primeiro verificar se há uma transmissão ao vivo
    $live_api_url = "https://www.googleapis.com/youtube/v3/search?key={$api_key}&channelId={$channel_id}&part=snippet,id&eventType=live&type=video&maxResults=1";
    
    try {
        $live_response = @file_get_contents($live_api_url);
        
        if ($live_response !== false) {
            $live_data = json_decode($live_response, true);
            
            // Se encontrou uma transmissão ao vivo
            if (isset($live_data['items']) && !empty($live_data['items'])) {
                $video = $live_data['items'][0];
                $video_id = $video['id']['videoId'] ?? '';
                $title = $video['snippet']['title'] ?? '';
                $thumbnail = $video['snippet']['thumbnails']['high']['url'] ?? '';
                $published_at = $video['snippet']['publishedAt'] ?? '';
                
                return [
                    'success' => true,
                    'video_id' => $video_id,
                    'title' => $title,
                    'thumbnail' => $thumbnail,
                    'published_at' => $published_at,
                    'url' => "https://www.youtube.com/watch?v={$video_id}",
                    'is_live' => true
                ];
            }
        }
        
        // Se não encontrou transmissão ao vivo, buscar o último vídeo
        $api_url = "https://www.googleapis.com/youtube/v3/search?key={$api_key}&channelId={$channel_id}&part=snippet,id&order=date&maxResults=1";
        
        $response = @file_get_contents($api_url);
        
        if ($response === false) {
            return [
                'success' => false,
                'message' => 'Erro ao acessar a API do YouTube'
            ];
        }
        
        $data = json_decode($response, true);
        
        if (!isset($data['items']) || empty($data['items'])) {
            return [
                'success' => false,
                'message' => 'Nenhum vídeo encontrado'
            ];
        }
        
        $video = $data['items'][0];
        $video_id = $video['id']['videoId'] ?? '';
        $title = $video['snippet']['title'] ?? '';
        $thumbnail = $video['snippet']['thumbnails']['high']['url'] ?? '';
        $published_at = $video['snippet']['publishedAt'] ?? '';
        
        return [
            'success' => true,
            'video_id' => $video_id,
            'title' => $title,
            'thumbnail' => $thumbnail,
            'published_at' => $published_at,
            'url' => "https://www.youtube.com/watch?v={$video_id}",
            'is_live' => false
        ];
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Erro: ' . $e->getMessage()
        ];
    }
}

// Função para exibir múltiplos banners
function displayMultiBanners($banners) {
    if (empty($banners)) {
        return '<div class="text-center text-gray-400 py-4"></div>';
    }
    
    $count = count($banners);
    $grid_class = 'grid-cols-1';
    
    if ($count === 2) {
        $grid_class = 'grid-cols-1 md:grid-cols-2';
    } elseif ($count === 3) {
        $grid_class = 'grid-cols-1 md:grid-cols-3';
    }
    
    $html = '<div class="grid ' . $grid_class . ' gap-4">';
    
    foreach ($banners as $banner) {
        $html .= '
        <a href="' . htmlspecialchars($banner['link_url']) . '" 
           target="_blank" 
           class="banner-card group" 
           data-banner-id="' . $banner['id'] . '">
            <div class="relative overflow-hidden rounded-lg shadow-lg transition-all duration-300 transform group-hover:scale-105">
                <img 
                    src="' . htmlspecialchars($banner['image_url']) . '" 
                    alt="' . htmlspecialchars($banner['title']) . '" 
                    class="w-full h-48 object-cover transition-all duration-500 group-hover:brightness-90"
                    onerror="this.src=\'https://via.placeholder.com/800x300?text=Banner\'"
                >
                <div class="absolute inset-0 bg-gradient-to-t from-black/70 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300 flex items-end">
                    <div class="p-4 w-full">
                        <h3 class="text-white font-bold">' . htmlspecialchars($banner['title']) . '</h3>
                    </div>
                </div>
            </div>
        </a>';
    }
    
    $html .= '</div>';
    
    return $html;
}

// Função para exibir o último vídeo do YouTube
function displayLatestYoutubeVideo($video) {
    if (!$video['success']) {
        return '<div class="text-center text-gray-400 py-4">Não foi possível carregar o último vídeo.</div>';
    }
    
    $liveTag = '';
    
    $html = '
    <div class="bg-secondary rounded-lg overflow-hidden shadow-lg">
        <a href="' . $video['url'] . '" target="_blank" class="block relative">
            ' . $liveTag . '
            <div class="relative pb-[56.25%] h-0 group">
                <iframe 
                    src="https://www.youtube.com/embed/' . $video['video_id'] . '" 
                    frameborder="0" 
                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" 
                    allowfullscreen
                    class="absolute top-0 left-0 w-full h-full"
                ></iframe>
                <div class="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-20 flex items-center justify-center transition-all duration-300">
                    <div class="bg-red-600 text-white rounded-full w-16 h-16 flex items-center justify-center opacity-0 group-hover:opacity-100 transform scale-50 group-hover:scale-100 transition-all duration-300">
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-8 w-8" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z" />
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                        </svg>
                    </div>
                </div>
            </div>
        </a>
        <div class="p-4">
            <a href="' . $video['url'] . '" target="_blank" class="block hover:text-accent transition-colors duration-300">
                <h3 class="text-lg font-semibold text-light">' . htmlspecialchars($video['title']) . '</h3>
            </a>
            <p class="text-sm text-gray-400 mt-1">
                ' . (isset($video['is_live']) && $video['is_live'] ? 'Ao vivo agora' : 'Publicado em: ' . date('d/m/Y', strtotime($video['published_at']))) . '
            </p>
        </div>
    </div>';
    
    return $html;
}

// Função para adicionar o JavaScript de rastreamento
function getTrackingScript() {
    return '
    <script>
        document.addEventListener("DOMContentLoaded", function() {
            // Rastrear visualização da página
            fetch("track_pageview.php", {
                method: "POST",
                headers: {
                    "Content-Type": "application/x-www-form-urlencoded",
                },
                body: "track_type=pageview"
            });
            
            // Rastrear cliques em links
            document.querySelectorAll(".link-card").forEach(function(link) {
                link.addEventListener("click", function(e) {
                    const linkId = this.getAttribute("data-link-id");
                    if (linkId) {
                        fetch("track_clicks.php", {
                            method: "POST",
                            headers: {
                                "Content-Type": "application/x-www-form-urlencoded",
                            },
                            body: "track_type=link&link_id=" + linkId
                        });
                    }
                });
            });
            
            // Rastrear cliques em banners
            document.querySelectorAll(".banner-card").forEach(function(banner) {
                banner.addEventListener("click", function(e) {
                    const bannerId = this.getAttribute("data-banner-id");
                    if (bannerId) {
                        fetch("track_clicks.php", {
                            method: "POST",
                            headers: {
                                "Content-Type": "application/x-www-form-urlencoded",
                            },
                            body: "track_type=banner&banner_id=" + bannerId
                        });
                    }
                });
            });
            
            // Rastrear cliques em redes sociais
            document.querySelectorAll(".social-icon").forEach(function(social) {
                social.addEventListener("click", function(e) {
                    const socialId = this.getAttribute("data-social-id");
                    if (socialId) {
                        fetch("track_clicks.php", {
                            method: "POST",
                            headers: {
                                "Content-Type": "application/x-www-form-urlencoded",
                            },
                            body: "track_type=social&social_id=" + socialId
                        });
                    }
                });
            });
        });
    </script>';
}
