<?php
require_once '../config.php';

// Verificar se o usuário está logado
if (!isLoggedIn()) {
    redirect('../login.php');
}

// Definir ícones predefinidos das principais redes sociais
$predefined_icons = [
    'instagram' => [
        'name' => 'Instagram',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M12 2c2.717 0 3.056.01 4.122.06 1.065.05 1.79.217 2.428.465.66.254 1.216.598 1.772 1.153.509.5.902 1.105 1.153 1.772.247.637.415 1.363.465 2.428.047 1.066.06 1.405.06 4.122 0 2.717-.01 3.056-.06 4.122-.05 1.065-.218 1.79-.465 2.428a4.883 4.883 0 0 1-1.153 1.772c-.5.508-1.105.902-1.772 1.153-.637.247-1.363.415-2.428.465-1.066.047-1.405.06-4.122.06-2.717 0-3.056-.01-4.122-.06-1.065-.05-1.79-.218-2.428-.465a4.89 4.89 0 0 1-1.772-1.153 4.904 4.904 0 0 1-1.153-1.772c-.248-.637-.415-1.363-.465-2.428C2.013 15.056 2 14.717 2 12c0-2.717.01-3.056.06-4.122.05-1.066.217-1.79.465-2.428a4.88 4.88 0 0 1 1.153-1.772A4.897 4.897 0 0 1 5.45 2.525c.638-.248 1.362-.415 2.428-.465C8.944 2.013 9.283 2 12 2zm0 1.802c-2.67 0-2.987.01-4.04.059-.977.045-1.504.207-1.857.344-.467.182-.8.398-1.15.748-.35.35-.566.683-.748 1.15-.137.353-.3.88-.344 1.857-.047 1.053-.059 1.37-.059 4.04 0 2.67.01 2.988.059 4.04.045.977.207 1.504.344 1.857.182.466.399.8.748 1.15.35.35.683.566 1.15.748.353.137.88.3 1.857.344 1.054.047 1.37.059 4.04.059 2.67 0 2.987-.01 4.04-.059.977-.045 1.504-.207 1.857-.344.467-.182.8-.398 1.15-.748.35-.35.566-.683.748-1.15.137-.353.3-.88.344-1.857.047-1.054.059-1.37.059-4.04 0-2.67-.01-2.987-.059-4.04-.045-.977-.207-1.504-.344-1.857a3.097 3.097 0 0 0-.748-1.15 3.098 3.098 0 0 0-1.15-.748c-.353-.137-.88-.3-1.857-.344-1.053-.047-1.37-.059-4.04-.059zm0 3.063A5.135 5.135 0 1 1 12.002 17.2 5.135 5.135 0 0 1 12 6.865zM12 15.333a3.333 3.333 0 1 0 0-6.666 3.333 3.333 0 0 0 0 6.666zm6.533-8.671a1.2 1.2 0 1 1-2.4 0 1.2 1.2 0 0 1 2.4 0z"/></svg>',
        'color' => '#E1306C'
    ],
    'facebook' => [
        'name' => 'Facebook',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/></svg>',
        'color' => '#1877F2'
    ],
    'twitter' => [
        'name' => 'Twitter / X',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M18.901 1.153h3.68l-8.04 9.19L24 22.846h-7.406l-5.8-7.584-6.638 7.584H.474l8.6-9.83L0 1.154h7.594l5.243 6.932ZM17.61 20.644h2.039L6.486 3.24H4.298Z"/></svg>',
        'color' => '#000000'
    ],
    'youtube' => [
        'name' => 'YouTube',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M23.498 6.186a3.016 3.016 0 0 0-2.122-2.136C19.505 3.545 12 3.545 12 3.545s-7.505 0-9.377.505A3.017 3.017 0 0 0 .502 6.186C0 8.07 0 12 0 12s0 3.93.502 5.814a3.016 3.016 0 0 0 2.122 2.136c1.871.505 9.376.505 9.376.505s7.505 0 9.377-.505a3.015 3.015 0 0 0 2.122-2.136C24 15.93 24 12 24 12s0-3.93-.502-5.814zM9.545 15.568V8.432L15.818 12l-6.273 3.568z"/></svg>',
        'color' => '#FF0000'
    ],
    'tiktok' => [
        'name' => 'TikTok',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M12.525.02c1.31-.02 2.61-.01 3.91-.02.08 1.53.63 3.09 1.75 4.17 1.12 1.11 2.7 1.62 4.24 1.79v4.03c-1.44-.05-2.89-.35-4.2-.97-.57-.26-1.1-.59-1.62-.93-.01 2.92.01 5.84-.02 8.75-.08 1.4-.54 2.79-1.35 3.94-1.31 1.92-3.58 3.17-5.91 3.21-1.43.08-2.86-.31-4.08-1.03-2.02-1.19-3.44-3.37-3.65-5.71-.02-.5-.03-1-.01-1.49.18-1.9 1.12-3.72 2.58-4.96 1.66-1.44 3.98-2.13 6.15-1.72.02 1.48-.04 2.96-.04 4.44-.99-.32-2.15-.23-3.02.37-.63.41-1.11 1.04-1.36 1.75-.21.51-.15 1.07-.14 1.61.24 1.64 1.82 3.02 3.5 2.87 1.12-.01 2.19-.66 2.77-1.61.19-.33.4-.67.41-1.06.1-1.79.06-3.57.07-5.36.01-4.03-.01-8.05.02-12.07z"/></svg>',
        'color' => '#000000'
    ],
    'linkedin' => [
        'name' => 'LinkedIn',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M20.447 20.452h-3.554v-5.569c0-1.328-.027-3.037-1.852-3.037-1.853 0-2.136 1.445-2.136 2.939v5.667H9.351V9h3.414v1.561h.046c.477-.9 1.637-1.85 3.37-1.85 3.601 0 4.267 2.37 4.267 5.455v6.286zM5.337 7.433a2.062 2.062 0 0 1-2.063-2.065 2.064 2.064 0 1 1 2.063 2.065zm1.782 13.019H3.555V9h3.564v11.452zM22.225 0H1.771C.792 0 0 .774 0 1.729v20.542C0 23.227.792 24 1.771 24h20.451C23.2 24 24 23.227 24 22.271V1.729C24 .774 23.2 0 22.222 0h.003z"/></svg>',
        'color' => '#0077B5'
    ],
    'pinterest' => [
        'name' => 'Pinterest',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M12.017 0C5.396 0 .029 5.367.029 11.987c0 5.079 3.158 9.417 7.618 11.162-.105-.949-.199-2.403.041-3.439.219-.937 1.406-5.957 1.406-5.957s-.359-.72-.359-1.781c0-1.663.967-2.911 2.168-2.911 1.024 0 1.518.769 1.518 1.688 0 1.029-.653 2.567-.992 3.992-.285 1.193.6 2.165 1.775 2.165 2.128 0 3.768-2.245 3.768-5.487 0-2.861-2.063-4.869-5.008-4.869-3.41 0-5.409 2.562-5.409 5.199 0 1.033.394 2.143.889 2.741.099.12.112.225.085.345-.09.375-.293 1.199-.334 1.363-.053.225-.172.271-.401.165-1.495-.69-2.433-2.878-2.433-4.646 0-3.776 2.748-7.252 7.92-7.252 4.158 0 7.392 2.967 7.392 6.923 0 4.135-2.607 7.462-6.233 7.462-1.214 0-2.354-.629-2.758-1.379l-.749 2.848c-.269 1.045-1.004 2.352-1.498 3.146 1.123.345 2.306.535 3.55.535 6.607 0 11.985-5.365 11.985-11.987C23.97 5.39 18.592.026 11.985.026L12.017 0z"/></svg>',
        'color' => '#E60023'
    ],
    'twitch' => [
        'name' => 'Twitch',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M11.571 4.714h1.715v5.143H11.57zm4.715 0H18v5.143h-1.714zM6 0L1.714 4.286v15.428h5.143V24l4.286-4.286h3.428L22.286 12V0zm14.571 11.143l-3.428 3.428h-3.429l-3 3v-3H6.857V1.714h13.714Z"/></svg>',
        'color' => '#9146FF'
    ],
    'telegram' => [
        'name' => 'Telegram',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M11.944 0A12 12 0 0 0 0 12a12 12 0 0 0 12 12 12 12 0 0 0 12-12A12 12 0 0 0 12 0a12 12 0 0 0-.056 0zm4.962 7.224c.1-.002.321.023.465.14a.506.506 0 0 1 .171.325c.016.093.036.306.02.472-.18 1.898-.962 6.502-1.36 8.627-.168.9-.499 1.201-.82 1.23-.696.065-1.225-.46-1.9-.902-1.056-.693-1.653-1.124-2.678-1.8-1.185-.78-.417-1.21.258-1.91.177-.184 3.247-2.977 3.307-3.23.007-.032.014-.15-.056-.212s-.174-.041-.249-.024c-.106.024-1.793 1.14-5.061 3.345-.48.33-.913.49-1.302.48-.428-.008-1.252-.241-1.865-.44-.752-.245-1.349-.374-1.297-.789.027-.216.325-.437.893-.663 3.498-1.524 5.83-2.529 6.998-3.014 3.332-1.386 4.025-1.627 4.476-1.635z"/></svg>',
        'color' => '#0088CC'
    ],
    'whatsapp' => [
        'name' => 'WhatsApp',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413Z"/></svg>',
        'color' => '#25D366'
    ],
    'discord' => [
        'name' => 'Discord',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M20.317 4.3698a19.7913 19.7913 0 00-4.8851-1.5152.0741.0741 0 00-.0785.0371c-.211.3753-.4447.8648-.6083 1.2495-1.8447-.2762-3.68-.2762-5.4868 0-.1636-.3933-.4058-.8742-.6177-1.2495a.077.077 0 00-.0785-.037 19.7363 19.7363 0 00-4.8852 1.515.0699.0699 0 00-.0321.0277C.5334 9.0458-.319 13.5799.0992 18.0578a.0824.0824 0 00.0312.0561c2.0528 1.5076 4.0413 2.4228 5.9929 3.0294a.0777.0777 0 00.0842-.0276c.4616-.6304.8731-1.2952 1.226-1.9942a.076.076 0 00-.0416-.1057c-.6528-.2476-1.2743-.5495-1.8722-.8923a.077.077 0 01-.0076-.1277c.1258-.0943.2517-.1923.3718-.2914a.0743.0743 0 01.0776-.0105c3.9278 1.7933 8.18 1.7933 12.0614 0a.0739.0739 0 01.0785.0095c.1202.099.246.1981.3728.2924a.077.077 0 01-.0066.1276 12.2986 12.2986 0 01-1.873.8914.0766.0766 0 00-.0407.1067c.3604.698.7719 1.3628 1.225 1.9932a.076.076 0 00.0842.0286c1.961-.6067 3.9495-1.5219 6.0023-3.0294a.077.077 0 00.0313-.0552c.5004-5.177-.8382-9.6739-3.5485-13.6604a.061.061 0 00-.0312-.0286zM8.02 15.3312c-1.1825 0-2.1569-1.0857-2.1569-2.419 0-1.3332.9555-2.4189 2.157-2.4189 1.2108 0 2.1757 1.0952 2.1568 2.419 0 1.3332-.9555 2.4189-2.1569 2.4189zm7.9748 0c-1.1825 0-2.1569-1.0857-2.1569-2.419 0-1.3332.9554-2.4189 2.1569-2.4189 1.2108 0 2.1757 1.0952 2.1568 2.419 0 1.3332-.946 2.4189-2.1568 2.4189Z"/></svg>',
        'color' => '#5865F2'
    ],
    'snapchat' => [
        'name' => 'Snapchat',
        'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="currentColor"><path d="M12.206.793c.99 0 4.347.276 5.93 3.821.529 1.193.403 3.219.299 4.847l-.003.06c-.012.18-.022.345-.03.51.075.045.203.09.401.09.3-.016.659-.12 1.033-.301.165-.088.344-.104.464-.104.182 0 .359.029.509.09.45.149.734.479.734.838.015.449-.39.839-1.213 1.168-.089.029-.209.075-.344.119-.45.135-1.139.36-1.333.81-.09.224-.061.524.12.868l.015.015c.06.136 1.526 3.475 4.791 4.014.255.044.435.27.42.509 0 .075-.015.149-.045.225-.24.569-1.273.988-3.146 1.271-.059.091-.12.375-.164.57-.029.179-.074.36-.134.553-.076.271-.27.405-.555.405h-.03c-.135 0-.313-.031-.538-.074-.36-.075-.765-.135-1.273-.135-.3 0-.599.015-.913.074-.6.104-1.123.464-1.723.884-.853.599-1.826 1.288-3.294 1.288-.06 0-.119-.015-.18-.015h-.149c-1.468 0-2.427-.675-3.279-1.288-.599-.42-1.107-.779-1.707-.884-.314-.045-.629-.074-.928-.074-.54 0-.958.089-1.272.149-.211.043-.391.074-.54.074-.374 0-.523-.224-.583-.42-.061-.192-.09-.36-.135-.553-.045-.195-.105-.42-.164-.575-1.918-.222-2.95-.642-3.189-1.226-.031-.075-.045-.165-.045-.239-.015-.225.15-.465.42-.509 3.264-.54 4.73-3.879 4.791-4.02l.016-.029c.18-.345.224-.645.119-.869-.195-.434-.884-.658-1.332-.809-.121-.029-.24-.074-.346-.119-1.107-.435-1.257-.93-1.197-1.273.09-.479.674-.793 1.168-.793.146 0 .27.029.383.074.42.194.789.3 1.104.3.234 0 .384-.06.465-.105l-.007-.514c-.105-1.636-.225-3.651.307-4.837C7.392 1.077 10.739.807 11.727.807l.419-.015h.06z"/></svg>',
        'color' => '#FFFC00'
    ]
];

// Processar exclusão
if (isset($_GET['delete']) && !empty($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    
    // Obter informações da rede social para excluir o ícone personalizado
    $sql = "SELECT custom_icon_url FROM socials WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $stmt->close();
    
    if ($result->num_rows === 1) {
        $social = $result->fetch_assoc();
        if (!empty($social['custom_icon_url'])) {
            $iconPath = '../' . $social['custom_icon_url'];
            if (file_exists($iconPath)) {
                unlink($iconPath);
            }
        }
    }
    
    $sql = "DELETE FROM socials WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $stmt->close();
    
    // Redirecionar para evitar reenvio do formulário
    redirect('socials.php?msg=deleted');
}

// Processar alteração de status
if (isset($_GET['toggle']) && !empty($_GET['toggle'])) {
    $id = (int)$_GET['toggle'];
    
    // Obter status atual
    $sql = "SELECT is_active FROM socials WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $stmt->close();
    
    if ($result->num_rows === 1) {
        $row = $result->fetch_assoc();
        $newStatus = $row['is_active'] ? 0 : 1;
        
        // Atualizar status
        $sql = "UPDATE socials SET is_active = ? WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ii", $newStatus, $id);
        $stmt->execute();
        $stmt->close();
    }
    
    // Redirecionar para evitar reenvio do formulário
    redirect('socials.php?msg=updated');
}

// Processar formulário de adição/edição
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $platform = sanitize($_POST['platform']);
    $url = sanitize($_POST['url']);
    $icon = sanitize($_POST['icon']);
    $order = (int)$_POST['order'];
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    $predefined_icon = isset($_POST['predefined_icon']) ? sanitize($_POST['predefined_icon']) : '';
    
    if (empty($platform) || empty($url)) {
        $error = 'Plataforma e URL são obrigatórios.';
    } else {
        $customIconUrl = '';
        $uploadOk = true;
        
        // Se um ícone predefinido foi selecionado, usar o SVG correspondente
        if (!empty($predefined_icon) && isset($predefined_icons[$predefined_icon])) {
            // Armazenar apenas o nome da rede social predefinida em vez do SVG completo
            $icon = 'predefined:' . $predefined_icon;
        }
        
        // Verificar se um ícone personalizado foi enviado
        if (isset($_FILES['custom_icon']) && $_FILES['custom_icon']['error'] === 0) {
            $uploadDir = '../uploads/socials/';
            
            // Criar diretório se não existir
            if (!file_exists($uploadDir)) {
                mkdir($uploadDir, 0777, true);
            }
            
            $fileName = basename($_FILES['custom_icon']['name']);
            $fileType = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
            $newFileName = 'icon-' . time() . '-' . rand(1000, 9999) . '.' . $fileType;
            $targetFile = $uploadDir . $newFileName;
            
            // Verificar se é uma imagem real
            $check = getimagesize($_FILES['custom_icon']['tmp_name']);
            if ($check === false) {
                $error = 'O arquivo não é uma imagem válida.';
                $uploadOk = false;
            }
            
            // Verificar o tamanho do arquivo (máximo 1MB)
            if ($_FILES['custom_icon']['size'] > 1000000) {
                $error = 'O arquivo é muito grande. Tamanho máximo: 1MB.';
                $uploadOk = false;
            }
            
            // Permitir apenas certos formatos de arquivo
            if ($fileType != "jpg" && $fileType != "png" && $fileType != "jpeg" && $fileType != "gif" && $fileType != "webp" && $fileType != "svg") {
                $error = 'Apenas arquivos JPG, JPEG, PNG, GIF, WEBP e SVG são permitidos.';
                $uploadOk = false;
            }
            
            // Fazer upload do arquivo
            if ($uploadOk) {
                if (move_uploaded_file($_FILES['custom_icon']['tmp_name'], $targetFile)) {
                    $customIconUrl = 'uploads/socials/' . $newFileName;
                } else {
                    $error = 'Ocorreu um erro ao fazer upload do ícone.';
                    $uploadOk = false;
                }
            }
        }
        
        if (!isset($error)) {
            if (isset($_POST['id']) && !empty($_POST['id'])) {
                // Atualizar rede social existente
                $id = (int)$_POST['id'];
                
                if (!empty($customIconUrl)) {
                    // Se um novo ícone foi enviado, excluir o antigo
                    $sql = "SELECT custom_icon_url FROM socials WHERE id = ?";
                    $stmt = $conn->prepare($sql);
                    $stmt->bind_param("i", $id);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    $stmt->close();
                    
                    if ($result->num_rows === 1) {
                        $social = $result->fetch_assoc();
                        if (!empty($social['custom_icon_url'])) {
                            $oldIconPath = '../' . $social['custom_icon_url'];
                            if (file_exists($oldIconPath)) {
                                unlink($oldIconPath);
                            }
                        }
                    }
                    
                    // Atualizar com novo ícone
                    $sql = "UPDATE socials SET platform = ?, url = ?, icon = ?, custom_icon_url = ?, order_position = ?, is_active = ? WHERE id = ?";
                    $stmt = $conn->prepare($sql);
                    $stmt->bind_param("ssssiis", $platform, $url, $icon, $customIconUrl, $order, $isActive, $id);
                } else {
                    // Manter o ícone atual
                    $sql = "UPDATE socials SET platform = ?, url = ?, icon = ?, order_position = ?, is_active = ? WHERE id = ?";
                    $stmt = $conn->prepare($sql);
                    $stmt->bind_param("sssiii", $platform, $url, $icon, $order, $isActive, $id);
                }
            } else {
                // Adicionar nova rede social
                $sql = "INSERT INTO socials (platform, url, icon, custom_icon_url, order_position, is_active) VALUES (?, ?, ?, ?, ?, ?)";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("ssssii", $platform, $url, $icon, $customIconUrl, $order, $isActive);
            }
            
            $stmt->execute();
            $stmt->close();
            
            // Redirecionar para evitar reenvio do formulário
            redirect('socials.php?msg=success');
        }
    }
}

// Obter rede social para edição
$editSocial = null;
if (isset($_GET['edit']) && !empty($_GET['edit'])) {
    $id = (int)$_GET['edit'];
    $sql = "SELECT * FROM socials WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $stmt->close();
    
    if ($result->num_rows === 1) {
        $editSocial = $result->fetch_assoc();
    }
}

// Obter todas as redes sociais
$sql = "SELECT * FROM socials ORDER BY order_position ASC";
$socials_result = $conn->query($sql);
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar Redes Sociais - Painel Administrativo</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <style>
        :root {
            --color-dark: #121212;
            --color-secondary: #1e1e1e;
            --color-primary: #6d28d9;
            --color-accent: #8b5cf6;
            --color-light: #f3f4f6;
        }
        body {
            background-color: var(--color-dark);
            color: var(--color-light);
            font-family: 'Inter', sans-serif;
        }
        .input {
            width: 100%;
            padding: 0.75rem;
            border-radius: 0.375rem;
            background-color: #2d2d2d;
            color: var(--color-light);
            border: 1px solid #3d3d3d;
        }
        .input:focus {
            outline: none;
            border-color: var(--color-accent);
        }
        .btn {
            padding: 0.75rem 1rem;
            border-radius: 0.375rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-primary {
            background-color: var(--color-primary);
            color: white;
        }
        .btn-primary:hover {
            background-color: var(--color-accent);
        }
        .btn-secondary {
            background-color: #2d2d2d;
            color: var(--color-light);
        }
        .btn-secondary:hover {
            background-color: #3d3d3d;
        }
        .social-icon-preview {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            background-color: var(--color-primary);
            color: white;
            margin-right: 10px;
        }
        .social-icon-preview svg {
            width: 24px;
            height: 24px;
        }
        .icon-option {
            display: flex;
            align-items: center;
            padding: 8px 12px;
            border-radius: 4px;
            cursor: pointer;
            transition: all 0.2s ease;
        }
        .icon-option:hover {
            background-color: #3d3d3d;
        }
        .icon-option.selected {
            background-color: var(--color-primary);
        }
        .icon-preview {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 12px;
        }
        .icon-preview svg {
            width: 20px;
            height: 20px;
        }
    </style>
</head>
<body>
    <div class="min-h-screen flex">
        <!-- Sidebar -->
        <?php include 'sidebar.php'; ?>
        
        <!-- Main Content -->
        <div class="flex-1 ml-64">
            <header class="bg-secondary p-4 shadow-md">
                <div class="flex justify-between items-center">
                    <h2 class="text-xl font-semibold text-light">Painel Administrativo</h2>
                    <div>
                        <a href="../index.php" target="_blank" class="btn btn-secondary text-sm">
                            Ver Página
                        </a>
                    </div>
                </div>
            </header>
            
            <main class="p-6">
                <div class="bg-secondary rounded-lg p-6 shadow-lg">
                    <div class="flex justify-between items-center mb-6">
                        <h1 class="text-2xl font-bold text-light">Gerenciar Redes Sociais</h1>
                        <?php if (!$editSocial): ?>
                            <button 
                                onclick="document.getElementById('formSection').classList.toggle('hidden')"
                                class="btn btn-primary"
                            >
                                Adicionar Rede Social
                            </button>
                        <?php endif; ?>
                    </div>
                    
                    <?php if (isset($_GET['msg'])): ?>
                        <div class="mb-4 p-3 rounded 
                            <?php echo $_GET['msg'] === 'deleted' ? 'bg-red-900/30 border border-red-800 text-red-300' : 'bg-green-900/30 border border-green-800 text-green-300'; ?>">
                            <?php 
                                if ($_GET['msg'] === 'success') echo 'Rede social salva com sucesso!';
                                elseif ($_GET['msg'] === 'deleted') echo 'Rede social excluída com sucesso!';
                                elseif ($_GET['msg'] === 'updated') echo 'Status da rede social atualizado com sucesso!';
                            ?>
                        </div>
                    <?php endif; ?>
                    
                    <?php if (isset($error)): ?>
                        <div class="mb-4 p-3 rounded bg-red-900/30 border border-red-800 text-red-300">
                            <?php echo $error; ?>
                        </div>
                    <?php endif; ?>
                    
                    <div id="formSection" class="<?php echo $editSocial ? '' : 'hidden'; ?> mb-8 bg-dark p-6 rounded-lg">
                        <h2 class="text-xl font-semibold text-light mb-4">
                            <?php echo $editSocial ? 'Editar Rede Social' : 'Nova Rede Social'; ?>
                        </h2>
                        <form method="POST" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" enctype="multipart/form-data" class="space-y-4">
                            <?php if ($editSocial): ?>
                                <input type="hidden" name="id" value="<?php echo $editSocial['id']; ?>">
                            <?php endif; ?>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-1">
                                    Plataforma
                                </label>
                                <input
                                    type="text"
                                    name="platform"
                                    class="input"
                                    value="<?php echo $editSocial ? htmlspecialchars($editSocial['platform']) : ''; ?>"
                                    required
                                    placeholder="Ex: Instagram, YouTube, Twitter"
                                />
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-1">
                                    URL
                                </label>
                                <input
                                    type="url"
                                    name="url"
                                    class="input"
                                    value="<?php echo $editSocial ? htmlspecialchars($editSocial['url']) : ''; ?>"
                                    required
                                    placeholder="https://"
                                />
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-1">
                                    Ícones Predefinidos
                                </label>
                                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-2 mb-4 max-h-60 overflow-y-auto p-2 bg-gray-800 rounded">
                                    <div class="icon-option" onclick="selectNone()">
                                        <div class="icon-preview bg-gray-700">
                                            <span>✖</span>
                                        </div>
                                        <span>Nenhum</span>
                                    </div>
                                    
                                    <?php foreach ($predefined_icons as $key => $icon_data): ?>
                                    <div class="icon-option" onclick="selectIcon('<?php echo $key; ?>', '<?php echo $icon_data['color']; ?>')">
                                        <div class="icon-preview" style="background-color: <?php echo $icon_data['color']; ?>">
                                            <?php echo $icon_data['icon']; ?>
                                        </div>
                                        <span><?php echo $icon_data['name']; ?></span>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                                <input type="hidden" name="predefined_icon" id="predefined_icon" value="">
                                <div id="selected-icon-preview" class="hidden mt-2 p-3 bg-gray-800 rounded flex items-center">
                                    <div id="preview-icon" class="social-icon-preview mr-3"></div>
                                    <div>
                                        <p class="font-medium" id="preview-name"></p>
                                        <p class="text-sm text-gray-400">Ícone selecionado</p>
                                    </div>
                                </div>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-1">
                                    Ícone (Emoji ou HTML)
                                </label>
                                <input
                                    type="text"
                                    name="icon"
                                    id="icon-input"
                                    class="input"
                                    value="<?php echo $editSocial ? htmlspecialchars($editSocial['icon']) : ''; ?>"
                                    placeholder="Emoji ou código do ícone (usado se não houver ícone predefinido ou personalizado)"
                                />
                                <p class="text-sm text-gray-400 mt-1">
                                    Este ícone será usado se não houver um ícone predefinido ou personalizado.
                                </p>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-1">
                                    Ícone Personalizado
                                </label>
                                <input
                                    type="file"
                                    name="custom_icon"
                                    class="input"
                                    accept="image/*"
                                />
                                <?php if ($editSocial && !empty($editSocial['custom_icon_url'])): ?>
                                    <div class="mt-2 flex items-center">
                                        <img 
                                            src="../<?php echo htmlspecialchars($editSocial['custom_icon_url']); ?>" 
                                            alt="Ícone atual" 
                                            class="w-10 h-10 object-contain rounded mr-2"
                                        >
                                        <p class="text-sm text-gray-400">
                                            Ícone atual. Envie uma nova imagem para substituí-lo ou deixe em branco para manter.
                                        </p>
                                    </div>
                                <?php endif; ?>
                                <p class="text-sm text-gray-400 mt-1">
                                    Tamanho máximo: 1MB. Formatos permitidos: JPG, PNG, GIF, WEBP, SVG.
                                </p>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-1">
                                    Ordem
                                </label>
                                <input
                                    type="number"
                                    name="order"
                                    class="input"
                                    value="<?php echo $editSocial ? (int)$editSocial['order_position'] : 0; ?>"
                                    min="0"
                                />
                            </div>
                            
                            <div class="flex items-center">
                                <input
                                    type="checkbox"
                                    id="is_active"
                                    name="is_active"
                                    class="mr-2"
                                    <?php echo (!$editSocial || $editSocial['is_active']) ? 'checked' : ''; ?>
                                >
                                <label for="is_active" class="text-sm font-medium text-gray-300">
                                    Ativo
                                </label>
                            </div>
                            
                            <div class="flex space-x-3">
                                <button type="submit" class="btn btn-primary">
                                    <?php echo $editSocial ? 'Atualizar' : 'Adicionar'; ?> Rede Social
                                </button>
                                <?php if ($editSocial): ?>
                                    <a href="socials.php" class="btn btn-secondary">
                                        Cancelar
                                    </a>
                                <?php else: ?>
                                    <button 
                                        type="button" 
                                        class="btn btn-secondary"
                                        onclick="document.getElementById('formSection').classList.add('hidden')"
                                    >
                                        Cancelar
                                    </button>
                                <?php endif; ?>
                            </div>
                        </form>
                    </div>
                    
                    <div class="bg-dark p-6 rounded-lg">
                        <h2 class="text-xl font-semibold text-light mb-4">Redes Sociais</h2>
                        
                        <?php if ($socials_result->num_rows > 0): ?>
                            <div class="overflow-x-auto">
                                <table class="w-full">
                                    <thead>
                                        <tr class="text-left border-b border-gray-700">
                                            <th class="pb-3 pr-4">Ícone</th>
                                            <th class="pb-3 pr-4">Plataforma</th>
                                            <th class="pb-3 pr-4">URL</th>
                                            <th class="pb-3 pr-4">Ordem</th>
                                            <th class="pb-3 pr-4">Status</th>
                                            <th class="pb-3">Ações</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while ($social = $socials_result->fetch_assoc()): ?>
                                            <tr class="border-b border-gray-800">
                                                <td class="py-3 pr-4">
                                                    <div class="social-icon-preview">
                                                        <?php if (!empty($social['custom_icon_url'])): ?>
                                                            <img 
                                                                src="../<?php echo htmlspecialchars($social['custom_icon_url']); ?>" 
                                                                alt="<?php echo htmlspecialchars($social['platform']); ?>" 
                                                                class="w-5 h-5 object-contain"
                                                            >
                                                        <?php else: ?>
                                                            <?php echo $social['icon']; ?>
                                                        <?php endif; ?>
                                                    </div>
                                                </td>
                                                <td class="py-3 pr-4"><?php echo htmlspecialchars($social['platform']); ?></td>
                                                <td class="py-3 pr-4">
                                                    <a 
                                                        href="<?php echo htmlspecialchars($social['url']); ?>" 
                                                        target="_blank"
                                                        class="text-accent hover:underline"
                                                    >
                                                        <?php echo htmlspecialchars($social['url']); ?>
                                                    </a>
                                                </td>
                                                <td class="py-3 pr-4"><?php echo (int)$social['order_position']; ?></td>
                                                <td class="py-3 pr-4">
                                                    <span class="px-2 py-1 rounded text-xs <?php echo $social['is_active'] ? 'bg-green-900/30 text-green-300' : 'bg-red-900/30 text-red-300'; ?>">
                                                        <?php echo $social['is_active'] ? 'Ativo' : 'Inativo'; ?>
                                                    </span>
                                                </td>
                                                <td class="py-3">
                                                    <div class="flex space-x-2">
                                                        <a 
                                                            href="?toggle=<?php echo $social['id']; ?>" 
                                                            class="px-2 py-1 bg-blue-900/30 text-blue-300 rounded text-xs hover:bg-blue-800/30"
                                                            title="<?php echo $social['is_active'] ? 'Desativar' : 'Ativar'; ?>"
                                                        >
                                                            <?php echo $social['is_active'] ? 'Desativar' : 'Ativar'; ?>
                                                        </a>
                                                        <a 
                                                            href="?edit=<?php echo $social['id']; ?>" 
                                                            class="px-2 py-1 bg-yellow-900/30 text-yellow-300 rounded text-xs hover:bg-yellow-800/30"
                                                        >
                                                            Editar
                                                        </a>
                                                        <a 
                                                            href="?delete=<?php echo $social['id']; ?>" 
                                                            class="px-2 py-1 bg-red-900/30 text-red-300 rounded text-xs hover:bg-red-800/30"
                                                            onclick="return confirm('Tem certeza que deseja excluir esta rede social?');"
                                                        >
                                                            Excluir
                                                        </a>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <p class="text-gray-400 text-center py-4">
                                Nenhuma rede social cadastrada. Adicione sua primeira rede social!
                            </p>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script>
        function selectIcon(key, color) {
            // Limpar seleções anteriores
            document.querySelectorAll('.icon-option').forEach(option => {
                option.classList.remove('selected');
            });
            
            // Selecionar o ícone clicado
            event.currentTarget.classList.add('selected');
            
            // Atualizar o campo oculto
            document.getElementById('predefined_icon').value = key;
            
            // Mostrar a pré-visualização
            const previewContainer = document.getElementById('selected-icon-preview');
            const previewIcon = document.getElementById('preview-icon');
            const previewName = document.getElementById('preview-name');
            
            previewContainer.classList.remove('hidden');
            previewIcon.style.backgroundColor = color;
            
            // Obter o nome e o ícone da opção selecionada
            const selectedOption = event.currentTarget;
            const iconName = selectedOption.querySelector('span:not(:first-child)').textContent;
            const iconSvg = selectedOption.querySelector('.icon-preview').innerHTML;
            
            previewIcon.innerHTML = iconSvg;
            previewName.textContent = iconName;
            
            // Limpar o campo de ícone manual
            document.getElementById('icon-input').value = '';
        }
        
        function selectNone() {
            // Limpar seleções
            document.querySelectorAll('.icon-option').forEach(option => {
                option.classList.remove('selected');
            });
            
            // Selecionar a opção "Nenhum"
            event.currentTarget.classList.add('selected');
            
            // Limpar o campo oculto
            document.getElementById('predefined_icon').value = '';
            
            // Esconder a pré-visualização
            document.getElementById('selected-icon-preview').classList.add('hidden');
        }
    </script>
</body>
</html>
