<?php
require_once '../config.php';

// Verificar se o usuário está logado
if (!isLoggedIn()) {
    redirect('../login.php');
}

// Obter as seções disponíveis
$sections = [];

// Verificar se a tabela sections existe
$sql = "SHOW TABLES LIKE 'sections'";
$result = $conn->query($sql);
$table_exists = ($result && $result->num_rows > 0);

// Se a tabela não existir, criar
if (!$table_exists) {
    $sql = "CREATE TABLE sections (
        section_id VARCHAR(50) PRIMARY KEY,
        name VARCHAR(100) NOT NULL,
        order_position INT NOT NULL DEFAULT 0
    )";
    
    if ($conn->query($sql) === TRUE) {
        // Inserir seções padrão
        $default_sections = [
            ['banner-section', 'Banners', 0],
            ['video-section', 'Último Vídeo', 1],
            ['links-section', 'Links', 2],
            ['social-section', 'Redes Sociais', 3]
        ];
        
        $insert_sql = "INSERT INTO sections (section_id, name, order_position) VALUES (?, ?, ?)";
        $stmt = $conn->prepare($insert_sql);
        
        foreach ($default_sections as $section) {
            $stmt->bind_param("ssi", $section[0], $section[1], $section[2]);
            $stmt->execute();
        }
        
        $stmt->close();
    } else {
        $error = "Erro ao criar tabela de seções: " . $conn->error;
    }
}

// Obter seções do banco de dados
$sql = "SELECT * FROM sections ORDER BY order_position ASC";
$result = $conn->query($sql);

if ($result && $result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $sections[] = $row;
    }
} else {
    // Seções padrão caso não existam no banco
    $sections = [
        ['section_id' => 'banner-section', 'name' => 'Banners', 'order_position' => 0],
        ['section_id' => 'video-section', 'name' => 'Último Vídeo', 'order_position' => 1],
        ['section_id' => 'links-section', 'name' => 'Links', 'order_position' => 2],
        ['section_id' => 'social-section', 'name' => 'Redes Sociais', 'order_position' => 3]
    ];
}

// Processar formulário de salvamento da ordem
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_order'])) {
    try {
        // Iniciar transação
        $conn->begin_transaction();
        
        // Atualizar a ordem de cada seção
        foreach ($_POST['section_order'] as $position => $section_id) {
            $position = (int)$position;
            $section_id = sanitize($section_id);
            
            $sql = "UPDATE sections SET order_position = ? WHERE section_id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("is", $position, $section_id);
            
            if (!$stmt->execute()) {
                throw new Exception("Erro ao atualizar a ordem da seção: " . $stmt->error);
            }
            
            $stmt->close();
        }
        
        // Confirmar transação
        $conn->commit();
        
        $message = "Ordem das seções salva com sucesso!";
        $message_type = "success";
        
        // Recarregar seções após salvar
        $sql = "SELECT * FROM sections ORDER BY order_position ASC";
        $result = $conn->query($sql);
        $sections = [];
        
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $sections[] = $row;
            }
        }
        
    } catch (Exception $e) {
        // Reverter transação em caso de erro
        $conn->rollback();
        
        $message = "Erro ao salvar a ordem das seções: " . $e->getMessage();
        $message_type = "error";
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Organização de Seções - Painel Administrativo</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        :root {
            --color-dark: #121212;
            --color-secondary: #1e1e1e;
            --color-primary: #6d28d9;
            --color-accent: #8b5cf6;
            --color-light: #f3f4f6;
        }
        body {
            background-color: var(--color-dark);
            color: var(--color-light);
            font-family: 'Inter', sans-serif;
        }
        .btn {
            padding: 0.75rem 1rem;
            border-radius: 0.375rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-primary {
            background-color: var(--color-primary);
            color: white;
        }
        .btn-primary:hover {
            background-color: var(--color-accent);
        }
        .btn-secondary {
            background-color: #2d2d2d;
            color: var(--color-light);
        }
        .btn-secondary:hover {
            background-color: #3d3d3d;
        }
        .sortable-item {
            cursor: move;
            transition: all 0.3s ease;
        }
        .sortable-item:hover {
            background-color: #2d2d2d;
        }
        .sortable-item.dragging {
            opacity: 0.5;
            background-color: #3d3d3d;
        }
        .drag-handle {
            cursor: move;
            color: #6d6d6d;
        }
        .drag-handle:hover {
            color: var(--color-accent);
        }
        .alert {
            padding: 1rem;
            border-radius: 0.375rem;
            margin-bottom: 1rem;
        }
        .alert-success {
            background-color: rgba(16, 185, 129, 0.2);
            border: 1px solid rgba(16, 185, 129, 0.4);
            color: #10b981;
        }
        .alert-error {
            background-color: rgba(239, 68, 68, 0.2);
            border: 1px solid rgba(239, 68, 68, 0.4);
            color: #ef4444;
        }
    </style>
</head>
<body>
    <div class="min-h-screen flex">
        <!-- Sidebar -->
        <?php include 'sidebar.php'; ?>
        
        <!-- Main Content -->
        <div class="flex-1 ml-64">
            <header class="bg-secondary p-4 shadow-md">
                <div class="flex justify-between items-center">
                    <h2 class="text-xl font-semibold text-light">Painel Administrativo</h2>
                    <div>
                        <a href="../index.php" target="_blank" class="btn btn-secondary text-sm">
                            Ver Página
                        </a>
                    </div>
                </div>
            </header>
            
            <main class="p-6">
                <div class="bg-secondary rounded-lg p-6 shadow-lg">
                    <h1 class="text-2xl font-bold text-light mb-6">Organização de Seções</h1>
                    
                    <?php if (!empty($message)): ?>
                        <div class="alert alert-<?php echo $message_type; ?>">
                            <?php echo $message; ?>
                        </div>
                    <?php endif; ?>
                    
                    <p class="text-gray-400 mb-6">
                        Arraste e solte as seções para reorganizá-las na página principal. A ordem será salva automaticamente.
                    </p>
                    
                    <form method="post" action="">
                        <div class="mb-6">
                            <ul id="sortable-sections" class="space-y-2">
                                <?php foreach ($sections as $index => $section): ?>
                                    <li class="sortable-item bg-dark p-4 rounded-lg flex items-center" data-id="<?php echo htmlspecialchars($section['section_id']); ?>">
                                        <span class="drag-handle mr-3">
                                            <i class="fas fa-grip-vertical"></i>
                                        </span>
                                        <span class="flex-1"><?php echo htmlspecialchars($section['name']); ?></span>
                                        <input type="hidden" name="section_order[]" value="<?php echo htmlspecialchars($section['section_id']); ?>">
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                        
                        <div class="flex justify-end">
                            <button type="submit" name="save_order" class="btn btn-primary">
                                Salvar Ordem
                            </button>
                        </div>
                    </form>
                    
                    <div class="mt-8 p-4 bg-dark rounded-lg">
                        <h3 class="text-lg font-semibold text-light mb-2">Visualização</h3>
                        <p class="text-gray-400 mb-4">
                            Veja como as seções aparecerão na página principal:
                        </p>
                        
                        <div class="space-y-2">
                            <?php foreach ($sections as $section): ?>
                                <div class="p-3 bg-secondary rounded border border-gray-700">
                                    <h4 class="font-medium"><?php echo htmlspecialchars($section['name']); ?></h4>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/sortablejs@1.14.0/Sortable.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const sortableList = document.getElementById('sortable-sections');
            
            if (sortableList) {
                new Sortable(sortableList, {
                    animation: 150,
                    handle: '.drag-handle',
                    ghostClass: 'bg-gray-700',
                    onEnd: function(evt) {
                        // Atualizar os inputs hidden com a nova ordem
                        const items = sortableList.querySelectorAll('.sortable-item');
                        const inputs = sortableList.querySelectorAll('input[name="section_order[]"]');
                        
                        items.forEach((item, index) => {
                            inputs[index].value = item.dataset.id;
                        });
                    }
                });
            }
        });
    </script>
</body>
</html>
