<?php
// Dashboard de métricas para a página bio
// Versão corrigida para resolver o problema de carregamento infinito

require_once '../config.php';

// Verificar se o usuário está logado
if (!isLoggedIn()) {
    redirect('../login.php');
}

// Obter período de filtro
$period = $_GET['period'] ?? 'week';
$custom_start = $_GET['start'] ?? '';
$custom_end = $_GET['end'] ?? '';

// Definir datas com base no período
$end_date = date('Y-m-d H:i:s');
$start_date = '';

switch ($period) {
    case 'today':
        $start_date = date('Y-m-d 00:00:00');
        $period_label = 'Hoje';
        break;
    case 'yesterday':
        $start_date = date('Y-m-d 00:00:00', strtotime('-1 day'));
        $end_date = date('Y-m-d 23:59:59', strtotime('-1 day'));
        $period_label = 'Ontem';
        break;
    case 'week':
        $start_date = date('Y-m-d 00:00:00', strtotime('-7 days'));
        $period_label = 'Últimos 7 dias';
        break;
    case 'month':
        $start_date = date('Y-m-d 00:00:00', strtotime('-30 days'));
        $period_label = 'Últimos 30 dias';
        break;
    case 'year':
        $start_date = date('Y-m-d 00:00:00', strtotime('-365 days'));
        $period_label = 'Último ano';
        break;
    case 'custom':
        if (!empty($custom_start) && !empty($custom_end)) {
            $start_date = date('Y-m-d 00:00:00', strtotime($custom_start));
            $end_date = date('Y-m-d 23:59:59', strtotime($custom_end));
            $period_label = 'Período personalizado: ' . date('d/m/Y', strtotime($custom_start)) . ' até ' . date('d/m/Y', strtotime($custom_end));
        } else {
            $start_date = date('Y-m-d 00:00:00', strtotime('-7 days'));
            $period_label = 'Últimos 7 dias';
        }
        break;
    default:
        $start_date = date('Y-m-d 00:00:00', strtotime('-7 days'));
        $period_label = 'Últimos 7 dias';
}

// Obter estatísticas gerais
$total_views = 0;
$total_link_clicks = 0;
$total_banner_clicks = 0;
$total_social_clicks = 0;

// Total de visualizações
$sql = "SELECT COUNT(*) as total FROM page_views WHERE visit_date BETWEEN ? AND ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();
if ($row = $result->fetch_assoc()) {
    $total_views = $row['total'];
}
$stmt->close();

// Total de cliques em links
$sql = "SELECT COUNT(*) as total FROM link_clicks WHERE click_date BETWEEN ? AND ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();
if ($row = $result->fetch_assoc()) {
    $total_link_clicks = $row['total'];
}
$stmt->close();

// Total de cliques em banners
$sql = "SELECT COUNT(*) as total FROM banner_clicks WHERE click_date BETWEEN ? AND ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();
if ($row = $result->fetch_assoc()) {
    $total_banner_clicks = $row['total'];
}
$stmt->close();

// Total de cliques em redes sociais
$sql = "SELECT COUNT(*) as total FROM social_clicks WHERE click_date BETWEEN ? AND ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();
if ($row = $result->fetch_assoc()) {
    $total_social_clicks = $row['total'];
}
$stmt->close();

// Obter dados para gráfico de visualizações por dia
$views_by_day = [];
$sql = "";

switch ($period) {
    case 'today':
    case 'yesterday':
        $sql = "SELECT 
                    HOUR(visit_date) as hour, 
                    COUNT(*) as total 
                FROM page_views 
                WHERE visit_date BETWEEN ? AND ? 
                GROUP BY HOUR(visit_date) 
                ORDER BY hour";
        break;
    case 'week':
    case 'month':
    case 'custom':
        if (strtotime($end_date) - strtotime($start_date) <= 60*60*24*31) { // Menos de 31 dias
            $sql = "SELECT 
                        DATE(visit_date) as date, 
                        COUNT(*) as total 
                    FROM page_views 
                    WHERE visit_date BETWEEN ? AND ? 
                    GROUP BY DATE(visit_date) 
                    ORDER BY date";
        } else {
            $sql = "SELECT 
                        YEAR(visit_date) as year,
                        MONTH(visit_date) as month, 
                        COUNT(*) as total 
                    FROM page_views 
                    WHERE visit_date BETWEEN ? AND ? 
                    GROUP BY YEAR(visit_date), MONTH(visit_date) 
                    ORDER BY year, month";
        }
        break;
    case 'year':
        $sql = "SELECT 
                    YEAR(visit_date) as year,
                    MONTH(visit_date) as month, 
                    COUNT(*) as total 
                FROM page_views 
                WHERE visit_date BETWEEN ? AND ? 
                GROUP BY YEAR(visit_date), MONTH(visit_date) 
                ORDER BY year, month";
        break;
}

$stmt = $conn->prepare($sql);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();

$views_labels = [];
$views_data = [];

while ($row = $result->fetch_assoc()) {
    if (isset($row['hour'])) {
        $views_labels[] = $row['hour'] . 'h';
        $views_data[] = $row['total'];
    } elseif (isset($row['date'])) {
        $views_labels[] = date('d/m', strtotime($row['date']));
        $views_data[] = $row['total'];
    } elseif (isset($row['month'])) {
        $month_names = [
            1 => 'Jan', 2 => 'Fev', 3 => 'Mar', 4 => 'Abr', 
            5 => 'Mai', 6 => 'Jun', 7 => 'Jul', 8 => 'Ago', 
            9 => 'Set', 10 => 'Out', 11 => 'Nov', 12 => 'Dez'
        ];
        $views_labels[] = $month_names[$row['month']] . '/' . $row['year'];
        $views_data[] = $row['total'];
    }
}
$stmt->close();

// Obter dados para gráfico de dispositivos
$devices = [];
$sql = "SELECT 
            visitor_device_type, 
            COUNT(*) as total 
        FROM page_views 
        WHERE visit_date BETWEEN ? AND ? 
        GROUP BY visitor_device_type";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();

$device_labels = [];
$device_data = [];

while ($row = $result->fetch_assoc()) {
    $device_type = ucfirst($row['visitor_device_type']);
    $device_labels[] = $device_type;
    $device_data[] = $row['total'];
}
$stmt->close();

// Obter dados para gráfico de países
$countries = [];
$sql = "SELECT 
            visitor_country, 
            COUNT(*) as total 
        FROM page_views 
        WHERE visit_date BETWEEN ? AND ? 
        GROUP BY visitor_country 
        ORDER BY total DESC 
        LIMIT 10";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();

$country_labels = [];
$country_data = [];

while ($row = $result->fetch_assoc()) {
    $country_labels[] = $row['visitor_country'];
    $country_data[] = $row['total'];
}
$stmt->close();

// Obter dados para gráfico de cliques em links
$links = [];
$sql = "SELECT 
            l.id,
            l.title,
            COUNT(lc.id) as total 
        FROM links l
        LEFT JOIN link_clicks lc ON l.id = lc.link_id AND lc.click_date BETWEEN ? AND ?
        WHERE l.is_active = 1
        GROUP BY l.id
        ORDER BY total DESC
        LIMIT 10";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();

$link_labels = [];
$link_data = [];

while ($row = $result->fetch_assoc()) {
    $link_labels[] = $row['title'];
    $link_data[] = $row['total'];
}
$stmt->close();

// Obter dados para gráfico de navegadores
$browsers = [];
$sql = "SELECT 
            visitor_browser, 
            COUNT(*) as total 
        FROM page_views 
        WHERE visit_date BETWEEN ? AND ? 
        GROUP BY visitor_browser 
        ORDER BY total DESC 
        LIMIT 5";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();

$browser_labels = [];
$browser_data = [];

while ($row = $result->fetch_assoc()) {
    $browser_labels[] = $row['visitor_browser'];
    $browser_data[] = $row['total'];
}
$stmt->close();

// Obter dados para gráfico de sistemas operacionais
$os = [];
$sql = "SELECT 
            visitor_os, 
            COUNT(*) as total 
        FROM page_views 
        WHERE visit_date BETWEEN ? AND ? 
        GROUP BY visitor_os 
        ORDER BY total DESC 
        LIMIT 5";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();

$os_labels = [];
$os_data = [];

while ($row = $result->fetch_assoc()) {
    $os_labels[] = $row['visitor_os'];
    $os_data[] = $row['total'];
}
$stmt->close();

// Obter dados para gráfico de regiões
$regions = [];
$sql = "SELECT 
            visitor_region, 
            COUNT(*) as total 
        FROM page_views 
        WHERE visit_date BETWEEN ? AND ? AND visitor_country = 'Brazil'
        GROUP BY visitor_region 
        ORDER BY total DESC 
        LIMIT 10";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();

$region_labels = [];
$region_data = [];

while ($row = $result->fetch_assoc()) {
    $region_labels[] = $row['visitor_region'];
    $region_data[] = $row['total'];
}
$stmt->close();

// Converter arrays para JSON para uso nos gráficos
$views_labels_json = json_encode($views_labels);
$views_data_json = json_encode($views_data);
$device_labels_json = json_encode($device_labels);
$device_data_json = json_encode($device_data);
$country_labels_json = json_encode($country_labels);
$country_data_json = json_encode($country_data);
$link_labels_json = json_encode($link_labels);
$link_data_json = json_encode($link_data);
$browser_labels_json = json_encode($browser_labels);
$browser_data_json = json_encode($browser_data);
$os_labels_json = json_encode($os_labels);
$os_data_json = json_encode($os_data);
$region_labels_json = json_encode($region_labels);
$region_data_json = json_encode($region_data);
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard de Métricas - Painel Administrativo</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        :root {
            --color-dark: #1e1e1e;
            --color-secondary: #1e1e1e;
            --color-primary: #6d28d9;
            --color-accent: #8b5cf6;
            --color-light: #f3f4f6;
        }
        body {
            background-color: var(--color-dark);
            color: var(--color-light);
            font-family: 'Inter', sans-serif;
        }
        .bg-dark {
            background-color: var(--color-dark);
        }
        .bg-secondary {
            background-color: var(--color-secondary);
        }
        .bg-primary {
            background-color: var(--color-primary);
        }
        .text-light {
            color: var(--color-light);
        }
        .text-accent {
            color: var(--color-accent);
        }
        .btn {
            padding: 0.5rem 1rem;
            border-radius: 0.375rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-primary {
            background-color: var(--color-primary);
            color: white;
        }
        .btn-primary:hover {
            background-color: var(--color-accent);
        }
        .btn-secondary {
            background-color: #2d2d2d;
            color: var(--color-light);
        }
        .btn-secondary:hover {
            background-color: #3d3d3d;
        }
        .stat-card {
            background-color: var(--color-secondary);
            border-radius: 0.5rem;
            padding: 1.5rem;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
        }
        .chart-container {
            background-color: var(--color-secondary);
            border-radius: 0.5rem;
            padding: 1.5rem;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            position: relative;
            height: 300px; /* Altura fixa para todos os gráficos */
            margin-bottom: 20px;
        }
        .date-picker {
            background-color: #2d2d2d;
            color: var(--color-light);
            border: 1px solid #3d3d3d;
            border-radius: 0.375rem;
            padding: 0.5rem;
        }
    </style>
</head>
<body>
    <div class="min-h-screen flex">
        <!-- Sidebar -->
        <?php include 'sidebar.php'; ?>
        
        <!-- Main Content -->
        <div class="flex-1 ml-64">
            <header class="bg-secondary p-4 shadow-md">
                <div class="flex justify-between items-center">
                    <h2 class="text-xl font-semibold text-light">Dashboard de Métricas</h2>
                    <div>
                        <a href="../index.php" target="_blank" class="btn btn-secondary text-sm">
                            Ver Página
                        </a>
                    </div>
                </div>
            </header>
            
            <main class="p-6">
                <!-- Filtro de período -->
                <div class="bg-secondary rounded-lg p-4 shadow-lg mb-6">
                    <form method="GET" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" class="flex flex-wrap items-center gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-400 mb-1">Período</label>
                            <select name="period" class="bg-dark text-light border border-gray-700 rounded px-3 py-2" onchange="toggleCustomDates(this.value)">
                                <option value="today" <?php echo $period === 'today' ? 'selected' : ''; ?>>Hoje</option>
                                <option value="yesterday" <?php echo $period === 'yesterday' ? 'selected' : ''; ?>>Ontem</option>
                                <option value="week" <?php echo $period === 'week' ? 'selected' : ''; ?>>Últimos 7 dias</option>
                                <option value="month" <?php echo $period === 'month' ? 'selected' : ''; ?>>Últimos 30 dias</option>
                                <option value="year" <?php echo $period === 'year' ? 'selected' : ''; ?>>Último ano</option>
                                <option value="custom" <?php echo $period === 'custom' ? 'selected' : ''; ?>>Personalizado</option>
                            </select>
                        </div>
                        
                        <div id="custom-dates" class="flex gap-2" style="display: <?php echo $period === 'custom' ? 'flex' : 'none'; ?>;">
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Data inicial</label>
                                <input type="date" name="start" value="<?php echo $custom_start; ?>" class="date-picker">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-400 mb-1">Data final</label>
                                <input type="date" name="end" value="<?php echo $custom_end; ?>" class="date-picker">
                            </div>
                        </div>
                        
                        <div class="self-end">
                            <button type="submit" class="btn btn-primary">Filtrar</button>
                        </div>
                    </form>
                </div>
                
                <h3 class="text-xl font-semibold text-light mb-4">Estatísticas: <?php echo $period_label; ?></h3>
                
                <!-- Cards de estatísticas -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                    <div class="stat-card">
                        <h4 class="text-gray-400 text-sm font-medium">Visualizações da Página</h4>
                        <p class="text-3xl font-bold text-light mt-2"><?php echo number_format($total_views, 0, ',', '.'); ?></p>
                    </div>
                    
                    <div class="stat-card">
                        <h4 class="text-gray-400 text-sm font-medium">Cliques em Links</h4>
                        <p class="text-3xl font-bold text-light mt-2"><?php echo number_format($total_link_clicks, 0, ',', '.'); ?></p>
                    </div>
                    
                    <div class="stat-card">
                        <h4 class="text-gray-400 text-sm font-medium">Cliques em Banners</h4>
                        <p class="text-3xl font-bold text-light mt-2"><?php echo number_format($total_banner_clicks, 0, ',', '.'); ?></p>
                    </div>
                    
                    <div class="stat-card">
                        <h4 class="text-gray-400 text-sm font-medium">Cliques em Redes Sociais</h4>
                        <p class="text-3xl font-bold text-light mt-2"><?php echo number_format($total_social_clicks, 0, ',', '.'); ?></p>
                    </div>
                </div>
                
                <!-- Gráficos -->
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
                    <!-- Gráfico de visualizações por período -->
                    <div class="chart-container">
                        <h4 class="text-lg font-semibold text-light mb-4">Visualizações por Período</h4>
                        <canvas id="viewsChart"></canvas>
                    </div>
                    
                    <!-- Gráfico de dispositivos -->
                    <div class="chart-container">
                        <h4 class="text-lg font-semibold text-light mb-4">Dispositivos</h4>
                        <canvas id="devicesChart"></canvas>
                    </div>
                    
                    <!-- Gráfico de países -->
                    <div class="chart-container">
                        <h4 class="text-lg font-semibold text-light mb-4">Top 10 Países</h4>
                        <canvas id="countriesChart"></canvas>
                    </div>
                    
                    <!-- Gráfico de cliques em links -->
                    <div class="chart-container">
                        <h4 class="text-lg font-semibold text-light mb-4">Top 10 Links Mais Clicados</h4>
                        <canvas id="linksChart"></canvas>
                    </div>
                    
                    <!-- Gráfico de navegadores -->
                    <div class="chart-container">
                        <h4 class="text-lg font-semibold text-light mb-4">Navegadores</h4>
                        <canvas id="browsersChart"></canvas>
                    </div>
                    
                    <!-- Gráfico de sistemas operacionais -->
                    <div class="chart-container">
                        <h4 class="text-lg font-semibold text-light mb-4">Sistemas Operacionais</h4>
                        <canvas id="osChart"></canvas>
                    </div>
                    
                    <!-- Gráfico de regiões do Brasil -->
                    <div class="chart-container">
                        <h4 class="text-lg font-semibold text-light mb-4">Top 10 Regiões do Brasil</h4>
                        <canvas id="regionsChart"></canvas>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script>
        // Função para alternar a exibição das datas personalizadas
        function toggleCustomDates(value) {
            const customDatesDiv = document.getElementById('custom-dates');
            if (value === 'custom') {
                customDatesDiv.style.display = 'flex';
            } else {
                customDatesDiv.style.display = 'none';
            }
        }
        
        // Configuração de cores para os gráficos
        const chartColors = [
            '#6d28d9', '#8b5cf6', '#a78bfa', '#c4b5fd', '#ddd6fe',
            '#4f46e5', '#4338ca', '#3730a3', '#312e81', '#1e1b4b',
            '#7c3aed', '#6d28d9', '#5b21b6', '#4c1d95', '#2e1065'
        ];
        
        // Configuração global para os gráficos
        Chart.defaults.color = '#f3f4f6';
        Chart.defaults.font.family = "'Inter', sans-serif";
        
        // Verificar se há dados antes de criar os gráficos
        const hasViewsData = <?php echo !empty($views_data) ? 'true' : 'false'; ?>;
        const hasDeviceData = <?php echo !empty($device_data) ? 'true' : 'false'; ?>;
        const hasCountryData = <?php echo !empty($country_data) ? 'true' : 'false'; ?>;
        const hasLinkData = <?php echo !empty($link_data) ? 'true' : 'false'; ?>;
        const hasBrowserData = <?php echo !empty($browser_data) ? 'true' : 'false'; ?>;
        const hasOsData = <?php echo !empty($os_data) ? 'true' : 'false'; ?>;
        const hasRegionData = <?php echo !empty($region_data) ? 'true' : 'false'; ?>;
        
        // Função para exibir mensagem de "Sem dados" nos gráficos
        function showNoDataMessage(canvasId, message = 'Sem dados disponíveis para o período selecionado') {
            const ctx = document.getElementById(canvasId).getContext('2d');
            ctx.clearRect(0, 0, ctx.canvas.width, ctx.canvas.height);
            ctx.font = '14px Inter, sans-serif';
            ctx.textAlign = 'center';
            ctx.textBaseline = 'middle';
            ctx.fillStyle = '#f3f4f6';
            ctx.fillText(message, ctx.canvas.width / 2, ctx.canvas.height / 2);
        }
        
        // Gráfico de visualizações por período
        if (hasViewsData) {
            const viewsCtx = document.getElementById('viewsChart').getContext('2d');
            const viewsChart = new Chart(viewsCtx, {
                type: 'line',
                data: {
                    labels: <?php echo $views_labels_json; ?>,
                    datasets: [{
                        label: 'Visualizações',
                        data: <?php echo $views_data_json; ?>,
                        backgroundColor: 'rgba(109, 40, 217, 0.2)',
                        borderColor: '#6d28d9',
                        borderWidth: 2,
                        tension: 0.1,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: 'rgba(255, 255, 255, 0.1)'
                            }
                        },
                        x: {
                            grid: {
                                color: 'rgba(255, 255, 255, 0.1)'
                            }
                        }
                    }
                }
            });
        } else {
            showNoDataMessage('viewsChart');
        }
        
        // Gráfico de dispositivos
        if (hasDeviceData) {
            const devicesCtx = document.getElementById('devicesChart').getContext('2d');
            const devicesChart = new Chart(devicesCtx, {
                type: 'doughnut',
                data: {
                    labels: <?php echo $device_labels_json; ?>,
                    datasets: [{
                        data: <?php echo $device_data_json; ?>,
                        backgroundColor: chartColors,
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'right'
                        }
                    }
                }
            });
        } else {
            showNoDataMessage('devicesChart');
        }
        
        // Gráfico de países
        if (hasCountryData) {
            const countriesCtx = document.getElementById('countriesChart').getContext('2d');
            const countriesChart = new Chart(countriesCtx, {
                type: 'bar',
                data: {
                    labels: <?php echo $country_labels_json; ?>,
                    datasets: [{
                        label: 'Visualizações',
                        data: <?php echo $country_data_json; ?>,
                        backgroundColor: chartColors,
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: 'rgba(255, 255, 255, 0.1)'
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            }
                        }
                    }
                }
            });
        } else {
            showNoDataMessage('countriesChart');
        }
        
        // Gráfico de cliques em links
        if (hasLinkData) {
            const linksCtx = document.getElementById('linksChart').getContext('2d');
            const linksChart = new Chart(linksCtx, {
                type: 'bar',
                data: {
                    labels: <?php echo $link_labels_json; ?>,
                    datasets: [{
                        label: 'Cliques',
                        data: <?php echo $link_data_json; ?>,
                        backgroundColor: chartColors,
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    indexAxis: 'y',
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        x: {
                            beginAtZero: true,
                            grid: {
                                color: 'rgba(255, 255, 255, 0.1)'
                            }
                        },
                        y: {
                            grid: {
                                display: false
                            }
                        }
                    }
                }
            });
        } else {
            showNoDataMessage('linksChart');
        }
        
        // Gráfico de navegadores
        if (hasBrowserData) {
            const browsersCtx = document.getElementById('browsersChart').getContext('2d');
            const browsersChart = new Chart(browsersCtx, {
                type: 'pie',
                data: {
                    labels: <?php echo $browser_labels_json; ?>,
                    datasets: [{
                        data: <?php echo $browser_data_json; ?>,
                        backgroundColor: chartColors,
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'right'
                        }
                    }
                }
            });
        } else {
            showNoDataMessage('browsersChart');
        }
        
        // Gráfico de sistemas operacionais
        if (hasOsData) {
            const osCtx = document.getElementById('osChart').getContext('2d');
            const osChart = new Chart(osCtx, {
                type: 'pie',
                data: {
                    labels: <?php echo $os_labels_json; ?>,
                    datasets: [{
                        data: <?php echo $os_data_json; ?>,
                        backgroundColor: chartColors,
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'right'
                        }
                    }
                }
            });
        } else {
            showNoDataMessage('osChart');
        }
        
        // Gráfico de regiões do Brasil
        if (hasRegionData) {
            const regionsCtx = document.getElementById('regionsChart').getContext('2d');
            const regionsChart = new Chart(regionsCtx, {
                type: 'bar',
                data: {
                    labels: <?php echo $region_labels_json; ?>,
                    datasets: [{
                        label: 'Visualizações',
                        data: <?php echo $region_data_json; ?>,
                        backgroundColor: chartColors,
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: {
                                color: 'rgba(255, 255, 255, 0.1)'
                            }
                        },
                        x: {
                            grid: {
                                display: false
                            }
                        }
                    }
                }
            });
        } else {
            showNoDataMessage('regionsChart');
        }
    </script>
</body>
</html>
