<?php
require_once '../config.php';

// Verificar se o usuário está logado
if (!isLoggedIn()) {
    redirect('../login.php');
}

// Processar exclusão
if (isset($_GET['delete']) && !empty($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    
    // Obter informações do link para excluir a imagem
    $sql = "SELECT image_url FROM links WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $stmt->close();
    
    if ($result->num_rows === 1) {
        $link = $result->fetch_assoc();
        if (!empty($link['image_url'])) {
            $imagePath = '../' . $link['image_url'];
            if (file_exists($imagePath)) {
                unlink($imagePath);
            }
        }
    }
    
    $sql = "DELETE FROM links WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $stmt->close();
    
    // Redirecionar para evitar reenvio do formulário
    redirect('links.php?msg=deleted');
}

// Processar alteração de status
if (isset($_GET['toggle']) && !empty($_GET['toggle'])) {
    $id = (int)$_GET['toggle'];
    
    // Obter status atual
    $sql = "SELECT is_active FROM links WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $stmt->close();
    
    if ($result->num_rows === 1) {
        $row = $result->fetch_assoc();
        $newStatus = $row['is_active'] ? 0 : 1;
        
        // Atualizar status
        $sql = "UPDATE links SET is_active = ? WHERE id = ?";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("ii", $newStatus, $id);
        $stmt->execute();
        $stmt->close();
    }
    
    // Redirecionar para evitar reenvio do formulário
    redirect('links.php?msg=updated');
}

// Processar formulário de adição/edição
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = sanitize($_POST['title']);
    $url = sanitize($_POST['url']);
    $icon = sanitize($_POST['icon']);
    $order = (int)$_POST['order'];
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    
    if (empty($title) || empty($url)) {
        $error = 'Título e URL são obrigatórios.';
    } else {
        $imageUrl = '';
        $uploadOk = true;
        
        // Verificar se uma imagem foi enviada
        if (isset($_FILES['image']) && $_FILES['image']['error'] === 0) {
            $uploadDir = '../uploads/links/';
            
            // Criar diretório se não existir
            if (!file_exists($uploadDir)) {
                mkdir($uploadDir, 0777, true);
            }
            
            $fileName = basename($_FILES['image']['name']);
            $fileType = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
            $newFileName = 'link-' . time() . '-' . rand(1000, 9999) . '.' . $fileType;
            $targetFile = $uploadDir . $newFileName;
            
            // Verificar se é uma imagem real
            $check = getimagesize($_FILES['image']['tmp_name']);
            if ($check === false) {
                $error = 'O arquivo não é uma imagem válida.';
                $uploadOk = false;
            }
            
            // Verificar o tamanho do arquivo (máximo 2MB)
            if ($_FILES['image']['size'] > 2000000) {
                $error = 'O arquivo é muito grande. Tamanho máximo: 2MB.';
                $uploadOk = false;
            }
            
            // Permitir apenas certos formatos de arquivo
            if ($fileType != "jpg" && $fileType != "png" && $fileType != "jpeg" && $fileType != "gif" && $fileType != "webp") {
                $error = 'Apenas arquivos JPG, JPEG, PNG, GIF e WEBP são permitidos.';
                $uploadOk = false;
            }
            
            // Fazer upload do arquivo
            if ($uploadOk) {
                if (move_uploaded_file($_FILES['image']['tmp_name'], $targetFile)) {
                    $imageUrl = 'uploads/links/' . $newFileName;
                } else {
                    $error = 'Ocorreu um erro ao fazer upload da imagem.';
                    $uploadOk = false;
                }
            }
        }
        
        if (!isset($error)) {
            if (isset($_POST['id']) && !empty($_POST['id'])) {
                // Atualizar link existente
                $id = (int)$_POST['id'];
                
                if (!empty($imageUrl)) {
                    // Se uma nova imagem foi enviada, excluir a antiga
                    $sql = "SELECT image_url FROM links WHERE id = ?";
                    $stmt = $conn->prepare($sql);
                    $stmt->bind_param("i", $id);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    $stmt->close();
                    
                    if ($result->num_rows === 1) {
                        $link = $result->fetch_assoc();
                        if (!empty($link['image_url'])) {
                            $oldImagePath = '../' . $link['image_url'];
                            if (file_exists($oldImagePath)) {
                                unlink($oldImagePath);
                            }
                        }
                    }
                    
                    // Atualizar com nova imagem
                    $sql = "UPDATE links SET title = ?, url = ?, icon = ?, image_url = ?, order_position = ?, is_active = ? WHERE id = ?";
                    $stmt = $conn->prepare($sql);
                    $stmt->bind_param("sssssii", $title, $url, $icon, $imageUrl, $order, $isActive, $id);
                } else {
                    // Manter a imagem atual
                    $sql = "UPDATE links SET title = ?, url = ?, icon = ?, order_position = ?, is_active = ? WHERE id = ?";
                    $stmt = $conn->prepare($sql);
                    $stmt->bind_param("sssiii", $title, $url, $icon, $order, $isActive, $id);
                }
            } else {
                // Adicionar novo link
                $sql = "INSERT INTO links (title, url, icon, image_url, order_position, is_active) VALUES (?, ?, ?, ?, ?, ?)";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("ssssii", $title, $url, $icon, $imageUrl, $order, $isActive);
            }
            
            $stmt->execute();
            $stmt->close();
            
            // Redirecionar para evitar reenvio do formulário
            redirect('links.php?msg=success');
        }
    }
}

// Obter link para edição
$editLink = null;
if (isset($_GET['edit']) && !empty($_GET['edit'])) {
    $id = (int)$_GET['edit'];
    $sql = "SELECT * FROM links WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $stmt->close();
    
    if ($result->num_rows === 1) {
        $editLink = $result->fetch_assoc();
    }
}

// Obter todos os links
$sql = "SELECT * FROM links ORDER BY order_position ASC";
$links_result = $conn->query($sql);
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar Links - Painel Administrativo</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <style>
        :root {
            --color-dark: #121212;
            --color-secondary: #1e1e1e;
            --color-primary: #6d28d9;
            --color-accent: #8b5cf6;
            --color-light: #f3f4f6;
        }
        body {
            background-color: var(--color-dark);
            color: var(--color-light);
            font-family: 'Inter', sans-serif;
        }
        .input {
            width: 100%;
            padding: 0.75rem;
            border-radius: 0.375rem;
            background-color: #2d2d2d;
            color: var(--color-light);
            border: 1px solid #3d3d3d;
        }
        .input:focus {
            outline: none;
            border-color: var(--color-accent);
        }
        .btn {
            padding: 0.75rem 1rem;
            border-radius: 0.375rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-primary {
            background-color: var(--color-primary);
            color: white;
        }
        .btn-primary:hover {
            background-color: var(--color-accent);
        }
        .btn-secondary {
            background-color: #2d2d2d;
            color: var(--color-light);
        }
        .btn-secondary:hover {
            background-color: #3d3d3d;
        }
    </style>
</head>
<body>
    <div class="min-h-screen flex">
        <!-- Sidebar -->
        <?php include 'sidebar.php'; ?>
        
        <!-- Main Content -->
        <div class="flex-1 ml-64">
            <header class="bg-secondary p-4 shadow-md">
                <div class="flex justify-between items-center">
                    <h2 class="text-xl font-semibold text-light">Painel Administrativo</h2>
                    <div>
                        <a href="../index.php" target="_blank" class="btn btn-secondary text-sm">
                            Ver Página
                        </a>
                    </div>
                </div>
            </header>
            
            <main class="p-6">
                <div class="bg-secondary rounded-lg p-6 shadow-lg">
                    <div class="flex justify-between items-center mb-6">
                        <h1 class="text-2xl font-bold text-light">Gerenciar Links</h1>
                        <?php if (!$editLink): ?>
                            <button 
                                onclick="document.getElementById('formSection').classList.toggle('hidden')"
                                class="btn btn-primary"
                            >
                                Adicionar Link
                            </button>
                        <?php endif; ?>
                    </div>
                    
                    <?php if (isset($_GET['msg'])): ?>
                        <div class="mb-4 p-3 rounded 
                            <?php echo $_GET['msg'] === 'deleted' ? 'bg-red-900/30 border border-red-800 text-red-300' : 'bg-green-900/30 border border-green-800 text-green-300'; ?>">
                            <?php 
                                if ($_GET['msg'] === 'success') echo 'Link salvo com sucesso!';
                                elseif ($_GET['msg'] === 'deleted') echo 'Link excluído com sucesso!';
                                elseif ($_GET['msg'] === 'updated') echo 'Status do link atualizado com sucesso!';
                            ?>
                        </div>
                    <?php endif; ?>
                    
                    <?php if (isset($error)): ?>
                        <div class="mb-4 p-3 rounded bg-red-900/30 border border-red-800 text-red-300">
                            <?php echo $error; ?>
                        </div>
                    <?php endif; ?>
                    
                    <div id="formSection" class="<?php echo $editLink ? '' : 'hidden'; ?> mb-8 bg-dark p-6 rounded-lg">
                        <h2 class="text-xl font-semibold text-light mb-4">
                            <?php echo $editLink ? 'Editar Link' : 'Novo Link'; ?>
                        </h2>
                        <form method="POST" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']); ?>" enctype="multipart/form-data" class="space-y-4">
                            <?php if ($editLink): ?>
                                <input type="hidden" name="id" value="<?php echo $editLink['id']; ?>">
                            <?php endif; ?>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-1">
                                    Título
                                </label>
                                <input
                                    type="text"
                                    name="title"
                                    class="input"
                                    value="<?php echo $editLink ? htmlspecialchars($editLink['title']) : ''; ?>"
                                    required
                                />
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-1">
                                    URL
                                </label>
                                <input
                                    type="url"
                                    name="url"
                                    class="input"
                                    value="<?php echo $editLink ? htmlspecialchars($editLink['url']) : ''; ?>"
                                    required
                                />
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-1">
                                    Ícone
                                </label>
                                <input
                                    type="text"
                                    name="icon"
                                    class="input"
                                    value="<?php echo $editLink ? htmlspecialchars($editLink['icon']) : '🔗'; ?>"
                                    placeholder="Emoji ou código do ícone"
                                />
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-1">
                                    Miniatura (Imagem)
                                </label>
                                <input
                                    type="file"
                                    name="image"
                                    class="input"
                                    accept="image/*"
                                />
                                <?php if ($editLink && !empty($editLink['image_url'])): ?>
                                    <div class="mt-2 flex items-center">
                                        <img 
                                            src="../<?php echo htmlspecialchars($editLink['image_url']); ?>" 
                                            alt="Miniatura atual" 
                                            class="w-16 h-16 object-cover rounded mr-2"
                                        >
                                        <p class="text-sm text-gray-400">
                                            Miniatura atual. Envie uma nova imagem para substituí-la ou deixe em branco para manter.
                                        </p>
                                    </div>
                                <?php endif; ?>
                                <p class="text-sm text-gray-400 mt-1">
                                    Tamanho máximo: 2MB. Formatos permitidos: JPG, PNG, GIF, WEBP.
                                </p>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-300 mb-1">
                                    Ordem
                                </label>
                                <input
                                    type="number"
                                    name="order"
                                    class="input"
                                    value="<?php echo $editLink ? (int)$editLink['order_position'] : 0; ?>"
                                    min="0"
                                />
                            </div>
                            
                            <div class="flex items-center">
                                <input
                                    type="checkbox"
                                    id="is_active"
                                    name="is_active"
                                    class="mr-2"
                                    <?php echo (!$editLink || $editLink['is_active']) ? 'checked' : ''; ?>
                                />
                                <label for="is_active" class="text-sm font-medium text-gray-300">
                                    Ativo
                                </label>
                            </div>
                            
                            <div class="flex space-x-3">
                                <button type="submit" class="btn btn-primary">
                                    <?php echo $editLink ? 'Salvar Alterações' : 'Adicionar Link'; ?>
                                </button>
                                <a 
                                    href="links.php" 
                                    class="btn btn-secondary"
                                >
                                    Cancelar
                                </a>
                            </div>
                        </form>
                    </div>
                    
                    <div class="space-y-4">
                        <?php if ($links_result->num_rows === 0): ?>
                            <p class="text-gray-400 text-center py-8">Nenhum link cadastrado.</p>
                        <?php else: ?>
                            <?php while($link = $links_result->fetch_assoc()): ?>
                                <div 
                                    class="flex items-center justify-between p-4 rounded-lg <?php echo $link['is_active'] ? 'bg-dark' : 'bg-dark/50'; ?> border <?php echo $link['is_active'] ? 'border-gray-700' : 'border-gray-800'; ?>"
                                >
                                    <div class="flex items-center">
                                        <?php if (!empty($link['image_url'])): ?>
                                            <div class="w-12 h-12 rounded-lg overflow-hidden mr-4 bg-gray-800 flex-shrink-0">
                                                <img 
                                                    src="../<?php echo htmlspecialchars($link['image_url']); ?>" 
                                                    alt="<?php echo htmlspecialchars($link['title']); ?>" 
                                                    class="w-full h-full object-cover"
                                                    onerror="this.src='https://via.placeholder.com/48?text=Erro'"
                                                >
                                            </div>
                                        <?php else: ?>
                                            <div class="w-10 h-10 rounded-full bg-primary flex items-center justify-center mr-4 flex-shrink-0">
                                                <span class="text-white"><?php echo htmlspecialchars($link['icon']); ?></span>
                                            </div>
                                        <?php endif; ?>
                                        <div>
                                            <h3 class="font-medium <?php echo $link['is_active'] ? 'text-light' : 'text-gray-400'; ?>">
                                                <?php echo htmlspecialchars($link['title']); ?>
                                            </h3>
                                            <p class="text-sm text-gray-500 truncate max-w-md">
                                                <?php echo htmlspecialchars($link['url']); ?>
                                            </p>
                                        </div>
                                    </div>
                                    <div class="flex space-x-2">
                                        <a 
                                            href="?toggle=<?php echo $link['id']; ?>"
                                            class="p-2 text-gray-400 hover:text-light"
                                            title="<?php echo $link['is_active'] ? 'Desativar' : 'Ativar'; ?>"
                                        >
                                            <?php echo $link['is_active'] ? '👁️' : '👁️‍🗨️'; ?>
                                        </a>
                                        <a 
                                            href="?edit=<?php echo $link['id']; ?>"
                                            class="p-2 text-gray-400 hover:text-light"
                                            title="Editar"
                                        >
                                            ✏️
                                        </a>
                                        <a 
                                            href="?delete=<?php echo $link['id']; ?>"
                                            class="p-2 text-gray-400 hover:text-red-500"
                                            title="Excluir"
                                            onclick="return confirm('Tem certeza que deseja excluir este link?');"
                                        >
                                            🗑️
                                        </a>
                                    </div>
                                </div>
                            <?php endwhile; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>
</body>
</html>
